import os

import yaml

from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import DogAppComponent, FakeRootComponent
from mail.devpack.ctl.lib.main import wrap_main
from mail.devpack.lib.yhttp_app_service import YplatformAppHttpService
from mail.webmail_config.lib.make_config import make_config
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.tvmapi import TvmApi


class Hound(DogAppComponent):
    NAME = 'hound'
    DEPS = [Sharpei, TvmApi]

    def __init__(self, env, components):
        super(Hound, self).__init__(env, components, custom_path='hound')
        self.__pyremock_port = self.config[self.NAME]['pyremock_port']

    def _init_root(self):
        self._generate_config('development')

        replace = {
            'tvmapi_port': self.components[TvmApi].port,
            'sharpei_port': self.components[Sharpei].webserver_port(),
            'passport_api_url': "http://localhost:{}".format(self.__pyremock_port),
        }

        devpack = self.yhttp.format_config(resource.find('hound/devpack.yml'), **replace)
        helpers.write2file(devpack, os.path.join(self.config_path, 'config-devpack.yml'))

    def _generate_config(self, env):
        base = resource.find('hound/base.yml')
        service_yaml = yaml.safe_load(resource.find('hound/service.yaml'))
        common_yaml = yaml.safe_load(resource.find('webmail_config/common.yaml'))
        cfg = make_config(env, base, service_yaml, common_yaml, silent=True)
        config = self.yhttp.format_config(cfg)
        helpers.write2file(config, os.path.join(self.config_path, 'config.yml'))

    @classmethod
    def gen_config(cls, port_generator, config=None):
        return dict(
            pyremock_port=next(port_generator),
            **YplatformAppHttpService.gen_config(port_generator, config=config)
        )

    @property
    def service_ticket(self):
        return self.components[TvmApi].get_ticket('tests', self.NAME)

    def init_root(self):
        self.yhttp.init_root()

        helpers.write2file(resource.find('hound/tvm_secret'), os.path.join(self.secrets_path, 'tvm_secret'))

        helpers.write2file(resource.find('recognizer/dict.dict'), os.path.join(self.resources_path, 'dict.dict'))
        helpers.write2file(resource.find('recognizer/queryrec.dict'), os.path.join(self.resources_path, 'queryrec.dict'))
        helpers.write2file(resource.find('recognizer/queryrec.weights'), os.path.join(self.resources_path, 'queryrec.weights'))

        helpers.write2file(resource.find('attach_sid_aes_keys.json'), os.path.join(self.secrets_path, 'attach_sid_aes_keys.json'))
        helpers.write2file(resource.find('attach_sid_hmac_keys.json'), os.path.join(self.secrets_path, 'attach_sid_hmac_keys.json'))

        helpers.write2file(resource.find('macs_pg/query.conf'), os.path.join(self.config_path, 'query.conf'))

        with open('hound.properties-devpack', 'w') as f:
            f.write('hound.host=http://localhost\n')
            f.write('hound.port={0}\n'.format(self.webserver_port()))

        self._init_root()

    def start(self, put_pgpass_in_env=True):
        super(Hound, self).start(put_pgpass_in_env=put_pgpass_in_env)

    def pyremock_port(self):
        return self.__pyremock_port


class HoundTesting(Hound):
    NAME = 'hound_testing'
    DEPS = []

    def _init_root(self):
        devpack = self.yhttp.format_config(resource.find('hound/testing.yml'))
        helpers.write2file(devpack, os.path.join(self.config_path, 'config-devpack.yml'))

        self._generate_config('testing')
        self.yhttp.init_pgpass(self.get_root())

        with open('hound.properties-testing', 'w') as f:
            f.write('hound.host=http://localhost\n')
            f.write('hound.port={0}\n'.format(self.webserver_port()))

    def start(self):
        super(HoundTesting, self).start(put_pgpass_in_env=True)


class HoundService(FakeRootComponent):
    NAME = 'hound-service'
    DEPS = [Hound, HoundTesting]


def main():
    wrap_main(deps_root=HoundService)
