#pragma once

#include <ymod_webserver/server.h>
#include <ymod_webserver/request.h>

#include <mail_errors/error_code.h>
#include <mail/hound/include/internal/wmi/errors.h>

#include <mail/hound/include/internal/logger.h>
#include <boost/optional.hpp>

#include <mail/hound/include/internal/wmi/yplatform/reflection/error.h>

#include <util/generic/typetraits.h>


namespace hound::server {

struct Context final {
    enum class ContentType {
        textPlain,
        applicationJson
    };

    ymod_webserver::request_ptr request;
    ContextLogger logger;
    std::string method;
    ymod_webserver::response_ptr response_;
    std::map<std::string, std::string> cookies;
    std::vector<std::pair<std::string, std::string> > headers;
    std::optional<std::string> uid;

    explicit Context(const ymod_webserver::http::stream_ptr& stream);

    template <typename Out>
    Out getArgList(const std::string& name, Out out) const {
        return boost::copy(request->url.params.equal_range(name) | boost::adaptors::map_values, out);
    }
    void setHeader(const std::string& name, const std::string& value);

    void setUid(const std::string& u);

    void responsePong();
    void responseEmptyJson();

    template<class Obj, class ... Args>
    void response(const Obj& obj, Args&&... args) {
        responseImpl(toJson(obj, std::forward<Args>(args)...), ContentType::applicationJson, ymod_webserver::codes::ok);
    }

    void responseError(const boost::system::error_code& code,
                       const std::string& reason);
    void responseError(const mail_errors::error_code& code);

    void responseErrorOnException();
    void responseBadRequestException();
    void responseInternalErrorException();

    void responseInternalError(const boost::system::error_code& code, const std::string& reason);
    void responseBadRequest(const boost::system::error_code& code, const std::string& reason);
    void responseBadRequest(const mail_errors::error_code& code);

    boost::optional<std::string> getOptionalArg(const std::string& name) const;
    boost::optional<std::string> getOptionalHeader(const std::string& name) const;
    std::string getArg(const std::string& name) const;
    bool getArgument(const std::string& name, std::string& value) const;
    bool getBoolArg(const std::string& name) const;
    std::string getCookie(const std::string& name) const;
    std::string getRawBody() const;

    std::string testBuckets() const;
    std::string enabledTestBuckets() const;
    std::string clientType() const;
    std::string clientVersion() const;
    std::string userAgent() const;
    std::string requestId() const;
    std::string getRealIp() const;
    std::string connectionId() const;

private:
    void parseCookieHeader();
    void setCodeAndHeadersInResponse(ymod_webserver::codes::code code);

    void responseImpl(const std::string& body, ContentType contentType, ymod_webserver::codes::code code);
    void responseErrorImpl(const boost::system::error_code& code,
                           const std::string& reason,
                           ymod_webserver::codes::code httpCode);

    template<class Obj, class... Args>
    std::string toJson(Obj&& obj, Args&&... args) const {
        if constexpr (std::is_same_v<std::decay_t<Obj>, std::string> || std::is_same_v<std::decay_t<Obj>, const char*>) {
            static_assert(TDependentFalse<Obj>, "cannot answer with string, use another method");
        } else {
            return yamail::data::serialization::toJson(obj, std::forward<Args>(args)...).str();
        }
    }
};

using ContextPtr = std::shared_ptr<Context>;

}
