#pragma once

#include <mail/hound/include/internal/server/base.h>

namespace hound::server::handlers {

enum class UserStrategy {
    existingOnly,
    existingThenDeleted
};

using UserType = macs::pg::UserType;

class MacsExecuter {
public:
    MacsExecuter(UserStrategy userStrategy);

    template <typename RequestPtr, typename Getter, typename Executer>
    void execute(RequestPtr r, Getter getMetadata, Executer executeMacs) const {
        auto& request = *r;

        auto types = getUserTypes();
        for (auto typeIt = types.begin(); typeIt != types.end(); ++typeIt) {
            auto metadata = getMetadata(request, *typeIt);
            try {
                executeMacs(r, metadata);
                return;
            } catch (const boost::system::system_error& err) {
                if (err.code() != ::sharpei::client::Errors::UidNotFound
                        || std::next(typeIt) == types.end()) {
                    throw;
                }
            }
        }
    }

private:
    UserStrategy userStrategy;

    std::vector<UserType> getUserTypes() const;
};

class MacsHandler : public Base {
public:
    MacsHandler(ConfigPtr config, UserStrategy userStrategy = UserStrategy::existingOnly)
        : Base(config), userStrategy(userStrategy) {
    }
protected:
    typedef macs::ServicePtr MailMetadataPtr;
    void execute(RequestPtr r, boost::asio::yield_context yield) const override {
        MacsExecuter(userStrategy).execute(r,
                [&](auto& r, UserType userType){ return this->getMetadata(r, userType); },
                [&](auto& r, auto& m) { this->executeMacs(r, m, yield); }
        );
    }

    virtual ::macs::pg::QueryHandleStrategy defaultQueryStrategy() const {
        return ::macs::pg::readNoLagReplicaThenMasterThenReplica;
    }

    virtual void executeMacs(RequestPtr request, MailMetadataPtr metadata, boost::asio::yield_context yieldCtx) const = 0;

protected:
    MailMetadataPtr getMetadata(Request& request, UserType userType = UserType::existing) const;

private:
    UserStrategy userStrategy;

    std::string getUserId(const Request& request) const;
    ::macs::pg::QueryHandleStrategy getQueryStrategy(const Request& request) const;
    boost::optional<std::string> getServiceDbUser(const Request& request) const;
};

}
