#pragma once

#include <boost/type_traits.hpp>
#include <boost/utility.hpp>
#include <pgg/enumeration.h>
#include <macs/envelopes_repository.h>
#include <mail/hound/include/internal/wmi/errors.h>
#include <mail/hound/include/internal/config.h>
#include <mail/hound/include/internal/v2/dereference.h>
#include <yamail/expected.h>

namespace hound::server {

namespace error = libwmi::error;
using mail_errors::error_code;
using yamail::expected;
using yamail::bad_expected_access;
using yamail::make_unexpected;

class QueryParams {
public:
    template<class QP, class Request, class Mailbox>
    error_code getParams(QP& params, const Request& request, const Mailbox& mailbox) const {
        auto parsed = expected<QP>(QP())
                .bind(getPageParam<QP>(request))
                .bind(getTimestampRangeParam<QP>(request))
                .bind(getMidParam<QP>(request))
                .bind(getDeviationParam<QP>(request))
                .bind(getLabelParam<QP>(request, mailbox))
                .bind(getFolderParam<QP>(request, mailbox))
                .bind(getFolderOrInboxParam<QP>(request, mailbox))
                .bind(getFoldersParam<QP>(request, mailbox))
                .bind(getTidParam<QP>(request))
                .bind(getTidsParam<QP>(request))
                .bind(getLidsParam<QP>(request))
                .bind(getSortTypeParam<QP>(request))
                .bind(getTabParam<QP>(request, mailbox))
                .bind(getMessageFormatParam<QP>(request));
        try {
            std::swap(params, parsed.value());
            return error_code();
        } catch (const bad_expected_access<error_code>& e) {
            return e.error();
        }
    }

    struct Page {
        int first;
        int count;
    };

    struct TimestampRange {
        std::optional<std::pair<time_t, time_t>> timestampRange;
    };

    struct Mid {
        std::string mid;
    };

    struct Deviation {
        size_t deviation;
    };

    struct Label {
        macs::Label label;
    };

    struct Folder {
        macs::Folder folder;
    };

    struct FolderOrInbox {
        macs::Folder folder;
    };

    struct Folders {
        std::vector<macs::Folder> folders;
    };

    struct Tid {
        std::string tid;
    };

    struct Tids {
        macs::Tids tids;
    };

    struct Lids {
        macs::Lids lids;
    };

    struct SortType {
        macs::EnvelopesSorting sortType;
    };

    class MessageFormat {
        struct __Format{
            enum Enum {
                none,
                full,
                midStid
            };

            using Map = typename pgg::Enum2String<Enum>::Map;

            void fill(Map & map) const{
                boost::assign::insert(map)
                    ( none, "" )
                    ( full, "full" )
                    ( midStid, "mid_stid" );
            }

            using Filler = __Format;
        };

    public:
        using Format = pgg::Enumeration<__Format, __Format::none>;

        Format messageFormat;
    };

    struct Tab {
        macs::Tab tab;
    };

private:
    template <class QP>
    using getter = std::function<expected<QP>(QP)>;

    template <bool V, class QP>
    using Require = std::enable_if_t<V, getter<QP>>;

    template <class Param, class QP>
    static constexpr auto BaseOf = std::is_base_of<Param, QP>::value;

    static auto skip() {
        return [](auto&& qp) { return std::move(qp); };
    }

    template <class QP, class Request>
    Require<BaseOf<Page, QP>, QP> getPageParam(const Request& request) const {
        return [&](QP&& params) -> expected<QP> {
            error_code ec = this->getPageParamImpl(params, request);
            if (ec) {
                return make_unexpected(std::move(ec));
            }
            return std::move(params);
        };
    }

    template <class QP, class Request>
    Require<!BaseOf<Page, QP>, QP> getPageParam(const Request&) const {
        return skip();
    }

    template <class QP, class Request>
    Require<BaseOf<TimestampRange, QP>, QP> getTimestampRangeParam(const Request& request) const {
        return [&](QP&& params) -> expected<QP> {
            error_code ec = this->getTimestampRangeParamImpl (params, request);
            if (ec) {
                return make_unexpected(std::move(ec));
            }
            return std::move(params);
        };
    }

    template <class QP, class Request>
    Require<!BaseOf<TimestampRange, QP>, QP> getTimestampRangeParam(const Request&) const {
        return skip();
    }

    template <class QP, class Request>
    Require<!BaseOf<Mid, QP>, QP> getMidParam (const Request&) const {
        return skip();
    }

    template <class QP, class Request>
    Require<BaseOf<Mid, QP>, QP> getMidParam (const Request& request) const {
        return [&](QP&& params) -> expected<QP> {
            if (!request.getArgument("mid", params.mid)) {
                return make_unexpected(error_code{error::invalidArgument, "mid parameter is required"});
            }
            return std::move(params);
        };
    }

    template <class QP, class Request>
    Require<!BaseOf<Deviation, QP>, QP> getDeviationParam (const Request&) const {
        return skip();
    }

    template <class QP, class Request>
    Require<BaseOf<Deviation, QP>, QP> getDeviationParam (const Request& request) const {
        return [&](QP&& params) -> expected<QP> {
            std::string arg;
            if (request.getArgument("deviation", arg)) {
                try {
                    params.deviation = boost::lexical_cast<size_t>(arg);
                } catch (const boost::bad_lexical_cast &) {
                    return make_unexpected(error_code{error::invalidArgument, "bad format of deviation parameter"});
                }
            } else {
                return make_unexpected(error_code{error::invalidArgument, "deviation parameter is required"});
            }
            return std::move(params);
        };
    }

    template <class QP, class Request, class Mailbox>
    Require<BaseOf<Label, QP>, QP> getLabelParam (const Request& request, const Mailbox& mailbox) const {
        return [&](QP&& params) -> expected<QP> {
            std::string lid;
            if (!request.getArgument("lid", lid)) {
                return make_unexpected(error_code{error::invalidArgument, "lid parameter is required"});
            }
            params.label = mailbox.labels().at(lid);
            return std::move(params);
        };
    }

    template <class QP, class Request, class Mailbox>
    Require<!BaseOf<Label, QP>, QP> getLabelParam(const Request& , const Mailbox&) const {
        return skip();
    }

    template <class QP, class Request, class Mailbox>
    Require<BaseOf<Folder, QP>, QP> getFolderParam (const Request& request, const Mailbox& mailbox) const {
        return [&](QP&& params) -> expected<QP> {
            std::string fid;
            if (!request.getArgument("fid", fid)) {
                return make_unexpected(error_code{error::invalidArgument, "fid parameter is required"});
            }
            params.folder = mailbox.folders().at(fid);
            return std::move(params);
        };
    }

    template <class QP, class Request, class Mailbox>
    Require<!BaseOf<Folder, QP>, QP> getFolderParam(const Request&, const Mailbox&) const {
        return skip();
    }

    template <class QP, class Request, class Mailbox>
    Require<BaseOf<FolderOrInbox, QP>, QP> getFolderOrInboxParam (const Request& request, const Mailbox& mailbox) const {
        return [&](QP&& params) -> expected<QP> {
            std::string fid;
            params.folder = request.getArgument("fid", fid)
                ? mailbox.folders().at(fid)
                : mailbox.folders().at(macs::Folder::Symbol::inbox);
            return std::move(params);
        };
    }

    template <class QP, class Request, class Mailbox>
    Require<!BaseOf<FolderOrInbox, QP>, QP> getFolderOrInboxParam(const Request&, const Mailbox&) const {
        return skip();
    }


    template <class QP, class Request, class Mailbox>
    Require<BaseOf<Folders, QP>, QP> getFoldersParam (const Request& request, const Mailbox& mailbox) const {
        return [&](QP&& params) -> expected<QP> {
            macs::FidList fids;
            request.getArgList( "fid", std::back_inserter(fids) );
            if( fids.empty() ) {
                return make_unexpected(error_code{error::invalidArgument, "fid parameter is required"});
            }
            const auto allFolders = mailbox.folders();
            std::transform(fids.begin(), fids.end(), std::back_inserter(params.folders),
                           [&allFolders](auto& fid) { return allFolders.at(fid); });
            return std::move(params);
        };
    }

    template <class QP, class Request, class Mailbox>
    Require<!BaseOf<Folders, QP>, QP> getFoldersParam (const Request&, const Mailbox&) const {
        return skip();
    }

    template <class QP, class Request>
    Require<BaseOf<Tid, QP>, QP> getTidParam (const Request& request) const {
        return [&](QP&& params) -> expected<QP> {
            if (!request.getArgument("tid", params.tid)) {
                return make_unexpected(error_code{error::invalidArgument, "tid parameter is required"});
            }
            return std::move(params);
        };
    }

    template <class QP, class Request>
    Require<!BaseOf<Tid, QP>, QP> getTidParam(const Request&) const {
        return skip();
    }

    template <class QP, class Request>
    Require<BaseOf<Tids, QP>, QP> getTidsParam (const Request& request) const {
        return [&](QP&& params) -> expected<QP> {
            request.getArgList( "tid", std::back_inserter(params.tids) );
            if (params.tids.empty()) {
                return make_unexpected(error_code{error::invalidArgument, "tid parameter is required"});
            }
            return std::move(params);
        };
    }

    template <class QP, class Request>
    Require<!BaseOf<Tids, QP>, QP> getTidsParam(const Request&) const {
        return skip();
    }

    template <class QP, class Request>
    Require<BaseOf<Lids, QP>, QP> getLidsParam(const Request& request) const {
        return [&](QP&& params) -> expected<QP> {
            request.getArgList("lid", std::back_inserter(params.lids));
            if (params.lids.empty()) {
                return make_unexpected(error_code{error::invalidArgument, "lid parameters are required"});
            }

            return std::move(params);
        };
    }

    template <class QP, class Request>
    Require<!BaseOf<Lids, QP>, QP> getLidsParam(const Request&) const {
        return skip();
    }

    template <class QP, class Request>
    Require<BaseOf<SortType, QP>, QP> getSortTypeParam (const Request& request) const {
        return [&](QP&& params) -> expected<QP> {
            params.sortType = this->maximacsSortSelect(request.getArg("sort_type"));
            return std::move(params);
        };
    }

    template <class QP, class Request>
    Require<!BaseOf<SortType, QP>, QP> getSortTypeParam(const Request&) const {
        return skip();
    }

    template <class QP, class Request>
    Require<BaseOf<MessageFormat, QP>, QP> getMessageFormatParam (const Request& request) const {
        return [&](QP&& params) -> expected<QP> {
            auto format = request.getOptionalArg("format");
            if (!format) {
                params.messageFormat = MessageFormat::Format::none;
            } else {
                params.messageFormat = MessageFormat::Format::fromString(*format, std::nothrow);
                if (params.messageFormat == MessageFormat::Format::none) {
                    return make_unexpected(error_code{error::invalidArgument, "unknown format: " + *format} );
                }
            }
            return std::move(params);
        };
    }

    template <class QP, class Request>
    Require<!BaseOf<MessageFormat, QP>, QP> getMessageFormatParam (const Request&) const {
        return skip();
    }

    template <class QP, class Request, class Mailbox>
    Require<BaseOf<Tab, QP>, QP> getTabParam (const Request& request, const Mailbox& mailbox) const {
        return [&](QP&& params) -> expected<QP> {
            std::string tabName = request.getArg("tab");
            if( tabName.empty() ) {
                return make_unexpected(error_code{error::invalidArgument, "tab parameter is required"});
            }
            params.tab = mailbox.tabs().at(tabName);
            return std::move(params);
        };
    }

    template <class QP, class Request, class Mailbox>
    Require<!BaseOf<Tab, QP>, QP> getTabParam (const Request&, const Mailbox&) const {
        return skip();
    }


    macs::EnvelopesSorting maximacsSortSelect(const std::string& s) const {
        static const std::unordered_map<std::string_view, macs::EnvelopesSorting> map = {
            { "date_descending", {macs::SortingType_descending} },
            { "date_ascending", {macs::SortingType_ascending} },
        };
        const auto i = map.find(s);
        return i != map.end() ? i->second : macs::EnvelopesSorting{};
    }

    template <class Request>
    error_code getPageParamImpl (Page& params, const Request& request) const {
        std::string requestArgument;
        if (request.getArgument("count", requestArgument)) {
            try {
                params.count = boost::lexical_cast<int>(requestArgument);
            } catch (const boost::bad_lexical_cast &) {
                return error_code{error::invalidArgument, "bad format of count parameter"};
            }
        } else {
            return error_code{error::invalidArgument, "count parameter is required"};
        }

        if (request.getArgument("first", requestArgument)) {
            try {
                params.first = boost::lexical_cast<int>(requestArgument);
            } catch (const boost::bad_lexical_cast &) {
                return error_code{error::invalidArgument, "bad format of first parameter"};
            }
        } else if (request.getArgument("page", requestArgument)) {
            try {
                const auto page = boost::lexical_cast<int>(requestArgument);
                if(page < 1) {
                    return error_code{error::invalidArgument, "page must be greater than 0"};
                }
                params.first = params.count * (page - 1);
            } catch (const boost::bad_lexical_cast &) {
                return error_code{error::invalidArgument, "bad format of page parameter"};
            }
            params.count++;
        } else {
            return error_code{error::invalidArgument, "first or page parameter is required"};
        }
        return error_code();
    }

    template <class Request>
    std::optional<time_t> getTimestampParam(const std::string& name, const Request& request) const {
        std::optional<time_t> value = std::nullopt;
        std::string requestArgument;
        if (request.getArgument(name, requestArgument)) {
            try {
                value = std::make_optional<time_t>(boost::lexical_cast<time_t>(requestArgument));
            } catch (const boost::bad_lexical_cast&) {
                throw std::runtime_error("bad format of " + name + " parameter: " + requestArgument);
            }
        }
        return value;
    }

    template <class Request>
    error_code getTimestampRangeParamImpl(TimestampRange& params, const Request& request) const {
        try {
            std::optional<time_t> since = getTimestampParam("since", request);
            std::optional<time_t> till = getTimestampParam("till", request);

            if (since || till) {
                const time_t defaultValue = 0;
                params.timestampRange = std::make_optional(
                            std::make_pair(since.value_or(defaultValue),
                                           till.value_or(defaultValue))
                );
            }
        } catch (const std::runtime_error& er) {
            return error_code{error::invalidArgument, er.what()};
        }
        return error_code();
    }
};

} // namespace hound::server
