#pragma once

#include <mail/hound/include/internal/v2/dereference.h>
#include <macs_pg/changelog/change_type.h>
#include <macs_pg/service/service.h>
#include <macs_pg/service/factory.h>

namespace hound::v2::changelog {

template <typename Impl, typename Context>
class Mailbox {
    Impl impl;
    Context ctx;
public:
    Mailbox(Impl impl, Context ctx = Context{}) : impl(impl), ctx(ctx) {}

    template <typename Range, typename Out>
    void envelopes(Range&& mids, Out out) const {
        auto res = service().envelopes().getByIds({std::begin(mids), std::end(mids)}, ctx);
        boost::copy(res, out);
    }

    template <typename Range, typename Out>
    void labels(Range&& lids, Out out) const {
        const auto labels = service().labels().getAllLabels(ctx);
        getValues(labels, std::forward<Range>(lids), std::forward<Out>(out));
    }

    template <typename Range, typename Out>
    void folders(Range&& fids, Out&& out) const {
        const auto folders = service().folders().getAllFolders(ctx);
        getValues(folders, std::forward<Range>(fids), std::forward<Out>(out));
    }

    template <typename Range, typename Out>
    void tabs(Range&& tabTypes, Out&& out) const {
        const auto tabs = service().tabs().getAllTabs(ctx);
        getValues(tabs, std::forward<Range>(tabTypes), std::forward<Out>(out));
    }

    macs::LabelSet getLabelsDict() const {
        return service().labels().getAllLabels(ctx);
    }

    bool checkUidRevisionExists(macs::Revision r) const {
        return service().changelog().getChangeIdByRevision(r, ctx) != boost::none;
    }

    template <typename Out>
    void getChangelog(macs::Revision r, std::int64_t limit, Out out) const {
        boost::copy(service().changelog().getChangelog(r, limit, ctx), out);
    }

    template <typename Out>
    void getChangelogByType(macs::Revision r, std::int64_t limit,
                            const std::vector<macs::pg::ChangeType>& changeTypes,
                            Out out) const {
        boost::copy(service().changelog().getChangelogByType(r, limit, changeTypes, ctx), out);
    }

private:
    decltype(auto) service() const { return detail::deref(impl); }

    template <typename Dict, typename Range, typename Out>
    static void getValues(const Dict& dict, Range&& ids, Out&& out) {
        for (const auto& id : ids) {
            const auto i = dict.find(id);
            if (i != dict.end()) {
                *out++ = i->second;
            }
        }
    }
};

namespace sync {

struct MailboxGetter {
    macs::pg::ServiceFactoryPtr factory;
    Mailbox<macs::ServicePtr, io_result::sync_context> operator()(const macs::Uid& uid) const {
        return {factory->mailbox(uid), {}};
    }
};

} // namespace sync

namespace async {

using yield_context = decltype(macs::io::make_yield_context(std::declval<boost::asio::yield_context>()));

struct MailboxGetter {
    macs::pg::ServiceFactoryPtr factory;
    yield_context yield;
    Mailbox<macs::ServicePtr, yield_context> operator()(const macs::Uid& uid) const {
        return {factory->mailbox(uid), yield};
    }
};

} // namespace async

} // namespace hound::v2::changelog
