#pragma once

#include <yamail/expected.h>
#include <mail/hound/include/internal/v2/changes/change.h>
#include <mail/hound/include/internal/wmi/errors.h>
#include <macs_pg/changelog/change_type.h>
#include <pgg/enumeration.h>
#include <boost/assign.hpp>


namespace hound::v2::changelog {

struct __CheckRevisionPolicy{
    enum Enum {
        unknown,
        strict,
        loyal
    };

    typedef pgg::Enum2String<Enum>::Map Map;
    void fill(Map & map) const {
        boost::assign::insert(map)
            ( strict, "strict" )
            ( loyal, "loyal" );
    }
    typedef __CheckRevisionPolicy Filler;
};

typedef pgg::Enumeration<__CheckRevisionPolicy> CheckRevisionPolicy;

} // namespace hound::v2::changelog

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changelog), Response,
    (std::vector<hound::v2::changes::Change>, changes)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changelog), Request,
    (macs::Uid, uid)
    (macs::Revision, revision)
    (std::int64_t, max_count)
    (std::vector<macs::pg::ChangeType>, change_types)
    (hound::v2::changelog::CheckRevisionPolicy, check_revision_policy)
)

namespace hound::v2::changelog {

namespace error = libwmi::error;
using mail_errors::error_code;

template <typename MailboxGetter>
struct Method {
    MailboxGetter getMailbox;

    yamail::expected<Response> operator() (Request args) const {
        if (args.max_count <= 0) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "max_count can not be less or equal to 0"});
        }

        if (args.max_count > 1000) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "max_count can not be more than 1000"});
        }

        if (args.uid.empty()) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "uid is empty"});
        }

        const auto mailbox = getMailbox(args.uid);

        if (args.check_revision_policy == CheckRevisionPolicy::strict &&
            !mailbox.checkUidRevisionExists(args.revision)) {
            return yamail::make_unexpected(error_code{error::revisionNotFound, "revision has not been found in changelog"});
        }

        std::vector<macs::Change> changelog;
        if (!args.change_types.empty()) {
            mailbox.getChangelogByType(args.revision, args.max_count, args.change_types,
                                       std::back_inserter(changelog));
        } else {
            mailbox.getChangelog(args.revision, args.max_count, std::back_inserter(changelog));
        }

        Response out;
        composeChanges(mailbox, changelog, std::back_inserter(out.changes));
        return out;
    }
};

} // namespace hound::v2::changelog
