#pragma once

/**
 * Delta API item - change and related stuff
 */

#include <mail/hound/include/internal/v2/changes/changelog_to_delta.h>
#include <boost/range/algorithm_ext/push_back.hpp>
#include <boost/variant.hpp>
#include <boost/range/algorithm/transform.hpp>

namespace hound::v2::changes {

template <typename ... Ts>
struct BasicChange {
    using Value = boost::variant<delta::Value<Ts>...>;

    std::int64_t revision;
    std::string_view type;
    Value value;
};

using Change = BasicChange<
    delta::Store,
    delta::Update,
    delta::Delete,
    delta::Copy,
    delta::Move,
    delta::QuickSave,
    delta::ThreadsJoin,
    delta::LabelCreate,
    delta::LabelDelete,
    delta::LabelModify,
    delta::FolderCreate,
    delta::FolderDelete,
    delta::FolderModify,
    delta::TabCreate,
    delta::MoveToTab,
    delta::StickerCreate,
    delta::StickerRemove
>;

template <typename Delta>
inline std::optional<Change> makeChange(macs::Revision r, Delta&& v) {
    if (empty(v)) {
        return std::nullopt;
    }
    return Change{std::int64_t(r), tag(v), std::move(value(std::forward<Delta>(v)))};
}

template <typename Mailbox>
struct ChangeComposer {
    using ChangeType = macs::pg::ChangeType;
    using Factory = std::function<std::tuple<macs::Revision, changelog::Change>(const macs::Change&, std::vector<macs::Mid>&)>;

    template <typename ChangelogChange>
    struct FactoryImpl {
        std::tuple<macs::Revision, changelog::Change> operator() (const macs::Change& in, std::vector<macs::Mid>& mids) const {
            ChangelogChange change = changelog::parse<ChangelogChange>(in);
            changelogToDeltaPrepare(change, mids);
            return {in.revision(), change};
        }
    };

    static const auto& factoryMap() {
        static const std::map<ChangeType::Enum, Factory> map = {
            {ChangeType::store, FactoryImpl<changelog::Store>{}},
            {ChangeType::update, FactoryImpl<changelog::Update>{}},
            {ChangeType::delete_, FactoryImpl<changelog::Delete>{}},
            {ChangeType::move, FactoryImpl<changelog::Move>{}},
            {ChangeType::copy, FactoryImpl<changelog::Copy>{}},
            {ChangeType::threadsJoin, FactoryImpl<changelog::ThreadsJoin>{}},
            {ChangeType::labelCreate, FactoryImpl<changelog::LabelCreate>{}},
            {ChangeType::labelDelete, FactoryImpl<changelog::LabelDelete>{}},
            {ChangeType::labelModify, FactoryImpl<changelog::LabelModify>{}},
            {ChangeType::folderCreate, FactoryImpl<changelog::FolderCreate>{}},
            {ChangeType::folderDelete, FactoryImpl<changelog::FolderDelete>{}},
            {ChangeType::folderModify, FactoryImpl<changelog::FolderModify>{}},
            {ChangeType::folderModifyType, FactoryImpl<changelog::FolderModify>{}},
            {ChangeType::quickSave, FactoryImpl<changelog::QuickSave>{}},
            {ChangeType::syncStore, FactoryImpl<changelog::Store>{}},
            {ChangeType::syncDelete, FactoryImpl<changelog::Delete>{}},
            {ChangeType::syncUpdate, FactoryImpl<changelog::Update>{}},
            {ChangeType::syncThreadsJoin, FactoryImpl<changelog::ThreadsJoin>{}},
            {ChangeType::tabCreate, FactoryImpl<changelog::TabCreate>{}},
            {ChangeType::moveToTab, FactoryImpl<changelog::MoveToTab>{}},
            {ChangeType::stickerCreate, FactoryImpl<changelog::StickerCreate>{}},
            {ChangeType::stickerRemove, FactoryImpl<changelog::StickerRemove>{}},
        };
        return map;
    }

    ChangeComposer(const Mailbox& mailbox) : mailbox_(&mailbox) {
    }

    static void prepare(const macs::Change& in, std::vector<macs::Mid>& mids, std::vector<std::tuple<macs::Revision,
            changelog::Change>>& changes) {

        const auto i = factoryMap().find(in.type());
        if (i != factoryMap().end()) {
            changes.push_back(i->second(in, mids));
        }
    }

    using MidToEnvelope = std::unordered_map<macs::Mid, macs::Envelope>;

    MidToEnvelope fetch(const std::vector<macs::Mid>& mids) const {
        MidToEnvelope result;
        if (!mids.empty()) {
            std::vector<macs::Envelope> envelopes;
            mailbox_->envelopes(mids, std::back_inserter(envelopes));

            for (auto& e : envelopes) {
                result[e.mid()] = std::move(e);
            }
        }
        return result;
    }

    template <typename Out>
    void make(const std::vector<std::tuple<macs::Revision, changelog::Change>>& changes,
            const MidToEnvelope& envelopes, Out&& out) const {
        for (auto& [revision, change]: changes) {
            std::optional<Change> ch = std::visit([&envelopes, revision = revision, this] (auto&& change) {
                return makeChange(revision, changelogToDelta(change, *mailbox_, envelopes));
            }, change);
            if (ch) {
                *out++ = std::move(*ch);
            }
        }
    }

    template <typename Range, typename Out>
    void operator() (const Range& changelog, Out&& out) const {
        std::vector<macs::Mid> mids;
        std::vector<std::tuple<macs::Revision, changelog::Change>> changes;
        for (auto& c : changelog) {
            prepare(c, mids, changes);
        }

        make(changes, fetch(mids), out);
    }

    const Mailbox* mailbox_ = nullptr;
};

template <typename Mailbox>
inline auto makeChangeComposer(const Mailbox& m) {
    return ChangeComposer<Mailbox>(m);
}

template <typename Mailbox, typename Range, typename Out>
inline void composeChanges(const Mailbox& mailbox,
                            const Range& changelog, Out&& out) {
    auto compose = makeChangeComposer(mailbox);
    compose(changelog, out);
}

using ChangeType = macs::pg::ChangeType;
static std::multimap<std::string, ChangeType> changeTypesFromString =
{
    {delta::Store::Tag::value, ChangeType::store},
    {delta::Store::Tag::value, ChangeType::syncStore},
    {delta::Update::Tag::value, ChangeType::update},
    {delta::Update::Tag::value, ChangeType::syncUpdate},
    {delta::Delete::Tag::value, ChangeType::delete_},
    {delta::Delete::Tag::value, ChangeType::syncDelete},
    {delta::Copy::Tag::value, ChangeType::copy},
    {delta::Move::Tag::value, ChangeType::move},
    {delta::QuickSave::Tag::value, ChangeType::quickSave},
    {delta::ThreadsJoin::Tag::value, ChangeType::threadsJoin},
    {delta::ThreadsJoin::Tag::value, ChangeType::syncThreadsJoin},
    {delta::LabelCreate::Tag::value, ChangeType::labelCreate},
    {delta::LabelDelete::Tag::value, ChangeType::labelDelete},
    {delta::LabelModify::Tag::value, ChangeType::labelModify},
    {delta::FolderCreate::Tag::value, ChangeType::folderCreate},
    {delta::FolderDelete::Tag::value, ChangeType::folderDelete},
    {delta::FolderModify::Tag::value, ChangeType::folderModify},
    {delta::FolderModify::Tag::value, ChangeType::folderModifyType},
    {delta::TabCreate::Tag::value, ChangeType::tabCreate},
    {delta::MoveToTab::Tag::value, ChangeType::moveToTab},
    {delta::StickerCreate::Tag::value, ChangeType::stickerCreate},
    {delta::StickerRemove::Tag::value, ChangeType::stickerRemove},
};

inline std::pair<std::vector<ChangeType>, std::vector<std::string>> changeTypesFromStrings(
    const std::vector<std::string>& deltaTypes)
{
    std::vector<ChangeType> changeTypes;
    std::vector<std::string> badInputs;
    for (const auto& deltaType : deltaTypes) {
        auto [first, last] = changeTypesFromString.equal_range(deltaType);
        auto range = boost::make_iterator_range(first, last);
        if (range.empty()) {
            badInputs.emplace_back(deltaType);
        } else {
            boost::push_back(changeTypes, range | boost::adaptors::map_values);
        }
    }
    return {changeTypes, badInputs};
}

} // namespace hound::v2::changes

BOOST_FUSION_ADAPT_STRUCT(hound::v2::changes::Change,
    revision, type, value
)
