#pragma once

/**
 * This file is dedicated to changelog entries' data representation.
 */

#include <macs/envelope.h>
#include <macs_pg/changelog/change.h>
#include <yamail/data/deserialization/json_reader.h>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <boost/range/algorithm/for_each.hpp>

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), MidOnly,
    (std::int64_t, mid)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), LidOnly,
    (std::int32_t, lid)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), FidOnly,
    (std::int32_t, fid)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), TabTypeOnly,
    (std::string, tab)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), ShortChangedMessage,
    (std::int64_t, mid)
    (boost::optional<std::int64_t>, tid)
    (std::int32_t, fid)
    (bool, seen)
    (bool, recent)
    (bool, deleted)
    (std::vector<std::int32_t>, lids)
    (std::string, tab)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), StickerOnly,
    (std::int64_t, mid)
    (std::string, stickerType)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), Store,
    (std::vector<hound::v2::changes::changelog::MidOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), Update,
    (std::vector<hound::v2::changes::changelog::ShortChangedMessage>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), Delete,
    (std::vector<hound::v2::changes::changelog::MidOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), Copy,
    (std::vector<hound::v2::changes::changelog::MidOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), Move,
    (std::vector<hound::v2::changes::changelog::ShortChangedMessage>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), QuickSave,
    (std::vector<hound::v2::changes::changelog::MidOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), ThreadsJoin,
    (std::vector<hound::v2::changes::changelog::ShortChangedMessage>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), LabelCreate,
    (std::vector<hound::v2::changes::changelog::LidOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), LabelModify,
    (hound::v2::changes::changelog::LidOnly, arguments)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), LabelDelete,
    (std::vector<hound::v2::changes::changelog::LidOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), FolderCreate,
    (std::vector<hound::v2::changes::changelog::FidOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), FolderModify,
    (std::vector<hound::v2::changes::changelog::FidOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), FolderDelete,
    (std::vector<hound::v2::changes::changelog::FidOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), TabCreate,
    (std::vector<hound::v2::changes::changelog::TabTypeOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), MoveToTab,
    (std::vector<hound::v2::changes::changelog::ShortChangedMessage>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), StickerCreate,
    (std::vector<hound::v2::changes::changelog::StickerOnly>, changed)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes::changelog), StickerRemove,
    (std::vector<hound::v2::changes::changelog::StickerOnly>, changed)
)

namespace hound::v2::changes::changelog {

using Change = std::variant<
        Store,
        Update,
        Delete,
        Copy,
        Move,
        QuickSave,
        ThreadsJoin,
        LabelCreate,
        LabelDelete,
        LabelModify,
        FolderCreate,
        FolderDelete,
        FolderModify,
        TabCreate,
        MoveToTab,
        StickerCreate,
        StickerRemove
>;

template <typename T>
using arguments_type = decltype(std::declval<std::decay_t<T>&>().arguments);

template <typename, typename = boost::void_t<>>
struct has_arguments : std::false_type {};

template <typename T>
struct has_arguments<T, boost::void_t<decltype(std::declval<T&>().arguments)>> : std::true_type {};

template <typename T>
using changed_type = decltype(std::declval<std::decay_t<T>>().changed);

template <typename, typename = boost::void_t<>>
struct has_changed : std::false_type {};

template <typename T>
struct has_changed<T, boost::void_t<decltype(std::declval<T&>().changed)>> : std::true_type {};

template <typename, typename = void>
struct ParseArgumentsImpl;

template <typename T>
struct ParseArgumentsImpl<T, std::enable_if_t<has_arguments<T>::value>> {
    static void apply(const macs::Change& in, T& out) {
        if (!in.arguments()) {
            throw std::invalid_argument(
                std::string(__PRETTY_FUNCTION__) +
                " change without arguments, id:" + std::to_string(in.changeId())
            );
        }
        using yamail::data::deserialization::fromJson;
        fromJson(*in.arguments(), out.arguments);
    }
};

template <typename T>
struct ParseNoop {
    static void apply(const macs::Change&, T&) {}
};

template <typename T>
struct ParseArgumentsImpl<T, std::enable_if_t<!has_arguments<T>::value>> : ParseNoop<T> {};

template <typename T>
inline void parseArguments(const macs::Change& in, T& out) {
    ParseArgumentsImpl<T>::apply(in, out);
}

template <typename, typename = void>
struct ParseChangedImpl;

template <typename T>
struct ParseChangedImpl<T, std::enable_if_t<has_changed<T>::value>> {
    static void apply(const macs::Change& in, T& out) {
        if (!in.changed()) {
            throw std::invalid_argument(
                std::string(__PRETTY_FUNCTION__) +
                " change without changed, id:" + std::to_string(in.changeId())
            );
        }
        using yamail::data::deserialization::fromJson;
        fromJson(*in.changed(), out.changed);
    }
};

template <typename T>
struct ParseChangedImpl<T, std::enable_if_t<!has_changed<T>::value>> : ParseNoop<T> {};

template <typename T>
inline void parseChanged(const macs::Change& in, T& out) {
    ParseChangedImpl<T>::apply(in, out);
}

template <typename T>
inline void parse(const macs::Change& in, T& out) {
    parseArguments(in, out);
    parseChanged(in, out);
}

template <typename T>
inline T parse(const macs::Change& in) {
    T out;
    parse(in, out);
    return out;
}

using SyncStore = Store;

using SyncUpdate = Update;

using SyncDelete = Delete;

using SyncThreadsJoin = ThreadsJoin;

using FolderModifyType = FolderModify;

template <typename Range>
inline auto extractMids(Range&& in) {
    return in | boost::adaptors::transformed([](auto& v) {
        using std::to_string;
        return to_string(v.mid);
    });
}

template <typename Range>
inline auto extractLids(Range&& in) {
    return in | boost::adaptors::transformed([](auto& v) {
        using std::to_string;
        return to_string(v.lid);
    });
}

template <typename Range>
inline auto extractFids(Range&& in) {
    return in | boost::adaptors::transformed([](auto& v) {
        using std::to_string;
        return to_string(v.fid);
    });
}

template <typename Range>
inline auto extractTabTypes(Range&& in) {
    return in | boost::adaptors::transformed([](auto& v) {
        return v.tab;
    });
}

inline auto extractLids(const ShortChangedMessage& change, const macs::LabelSet& labelsDict) {
    std::vector<macs::Lid> lids;

    using Symbol = macs::Label::Symbol;
    const auto seq = std::make_tuple(
                std::make_tuple(change.seen, Symbol::seen_label),
                std::make_tuple(change.recent, Symbol::recent_label),
                std::make_tuple(change.deleted, Symbol::deleted_label));

    boost::fusion::for_each(seq, [&](auto& item) {
        if (auto f = std::get<bool>(item)) {
            lids.push_back(labelsDict.at(std::get<Symbol>(item)).lid());
        }
    });

    boost::transform(change.lids, std::back_inserter(lids), [](std::int32_t lid) {
        using std::to_string;
        return to_string(lid);
    });

    return lids;
}

} // namespace hound::v2::changes::changelog
