#pragma once

/**
 * This file contains changelog entries to delta API entries mapping.
 */
#include <type_traits>
#include <mail/hound/include/internal/v2/changes/changelog.h>
#include <mail/hound/include/internal/v2/changes/delta.h>
#include <boost/range/algorithm/copy.hpp>
#include <boost/type_traits/function_traits.hpp>

namespace hound::v2::changes {

template <typename T>
struct ChangelogToDeltaImpl {
    template <typename Mailbox>
    static void apply(const T&, const Mailbox&, const std::unordered_map<macs::Mid, macs::Envelope>&) {
        static_assert(std::is_void<T>::value, "no changelog to delta convertion defined for the type");
    }

    static void prepare(const T&, std::vector<macs::Mid>&) {
        static_assert(std::is_void<T>::value, "no changelog to delta convertion defined for the type");
    }
};

template <typename In, typename Mailbox>
inline decltype(auto) changelogToDelta(const In& in, const Mailbox& m, const std::unordered_map<macs::Mid, macs::Envelope>& envelopes) {
    using applyType = decltype(ChangelogToDeltaImpl<In>::template apply<Mailbox>);

    if constexpr (boost::function_traits<applyType>::arity == 2) {
        return ChangelogToDeltaImpl<In>::apply(in, m);
    } else {
        return ChangelogToDeltaImpl<In>::apply(in, m, envelopes);
    }
};

template <typename, typename = std::void_t<>>
struct has_prepare_method : std::false_type {};

template <typename T>
struct has_prepare_method<T, std::void_t<decltype(T::prepare)>> : std::true_type {};

template <typename In>
inline void changelogToDeltaPrepare(const In& in, std::vector<macs::Mid>& mids) {
    if constexpr (has_prepare_method<ChangelogToDeltaImpl<In>>::value) {
        ChangelogToDeltaImpl<In>::prepare(in, mids);
    }
};

template <typename Change, typename Delta>
struct GetEnvelopeByMid {
    template <typename Mailbox>
    static auto apply(const Change& in, const Mailbox&, const std::unordered_map<macs::Mid, macs::Envelope>& envelopes) {
        Delta out;
        for (const auto& mid : extractMids(in.changed)) {
            auto it = envelopes.find(mid);
            if (envelopes.end() != it) {
                out.value.push_back(it->second);
            }
        }
        return out;
    }

    static void prepare(const Change& in, std::vector<macs::Mid>& mids) {
        auto midsRange = extractMids(in.changed);
        mids.insert(mids.end(), std::begin(midsRange), std::end(midsRange));
    }
};

template <>
struct ChangelogToDeltaImpl<changelog::Store> :
    GetEnvelopeByMid<changelog::Store, delta::Store> {};

template <>
struct ChangelogToDeltaImpl<changelog::Update> {
    template <typename Mailbox>
    static auto apply(const changelog::Update& in, const Mailbox& mailbox) {
        delta::Update out;
        out.value.reserve(in.changed.size());
        boost::transform(in.changed, std::back_inserter(out.value), [&](auto& v) {
            return delta::Update::Item{std::to_string(v.mid), extractLids(v, mailbox.getLabelsDict())};
        });
        return out;
    }
};

template <>
struct ChangelogToDeltaImpl<changelog::Delete> {
    template <typename Mailbox>
    static auto apply(const changelog::Delete& in, const Mailbox&) {
        delta::Delete out;
        out.value.reserve(in.changed.size());
        boost::copy(extractMids(in.changed), std::back_inserter(out.value));
        return out;
    }
};

template <>
struct ChangelogToDeltaImpl<changelog::Copy> :
    GetEnvelopeByMid<changelog::Copy, delta::Copy> {};

template <>
struct ChangelogToDeltaImpl<changelog::Move> {
    template <typename Mailbox>
    static auto apply(const changelog::Move& in, const Mailbox& mailbox) {
        delta::Move out;
        out.value.reserve(in.changed.size());
        boost::transform(in.changed, std::back_inserter(out.value), [&](auto& v) {
            delta::Move::Item item;
            item.mid = std::to_string(v.mid);
            item.fid = std::to_string(v.fid);
            if (v.tid) {
                item.tid = std::to_string(*(v.tid));
            }
            item.labels = extractLids(v, mailbox.getLabelsDict());
            return item;
        });
        return out;
    }
};

template <>
struct ChangelogToDeltaImpl<changelog::QuickSave> :
    GetEnvelopeByMid<changelog::QuickSave, delta::QuickSave> {};

template <>
struct ChangelogToDeltaImpl<changelog::ThreadsJoin> {
    template <typename Mailbox>
    static auto apply(const changelog::ThreadsJoin& in, const Mailbox& mailbox) {
        delta::ThreadsJoin out;
        out.value.reserve(in.changed.size());
        boost::transform(in.changed, std::back_inserter(out.value), [&](auto& v) {
            delta::ThreadsJoin::Item item;
            item.mid = std::to_string(v.mid);
            if (v.tid) {
                item.tid = std::to_string(*(v.tid));
            }
            item.labels = extractLids(v, mailbox.getLabelsDict());
            return item;
        });
        return out;
    }
};

template <typename Change, typename Delta>
struct GetLabelByLid {
    template <typename Mailbox>
    static auto apply(const Change& in, const Mailbox& mailbox) {
        Delta out;
        out.value.reserve(in.changed.size());
        mailbox.labels(extractLids(in.changed), std::back_inserter(out.value));
        return out;
    }
};

template <>
struct ChangelogToDeltaImpl<changelog::LabelCreate> :
    GetLabelByLid<changelog::LabelCreate, delta::LabelCreate> {};

template <>
struct ChangelogToDeltaImpl<changelog::LabelModify> {
    template <typename Mailbox>
    static auto apply(const changelog::LabelModify& in, const Mailbox& mailbox) {
        delta::LabelModify out;
        mailbox.labels(std::array<macs::Lid, 1>{{std::to_string(in.arguments.lid)}},
                            std::back_inserter(out.value));
        return out;
    }
};

template <>
struct ChangelogToDeltaImpl<changelog::LabelDelete> {
    template <typename Mailbox>
    static auto apply(const changelog::LabelDelete& in, const Mailbox&) {
        delta::LabelDelete out;
        out.value.reserve(in.changed.size());
        boost::copy(extractLids(in.changed), std::back_inserter(out.value));
        return out;
    }
};

template <typename Change, typename Delta>
struct GetFolderByFid {
    template <typename Mailbox>
    static auto apply(const Change& in, const Mailbox& mailbox) {
        Delta out;
        out.value.reserve(in.changed.size());
        mailbox.folders(extractFids(in.changed), std::back_inserter(out.value));
        return out;
    }
};

template <typename Change, typename Delta>
struct GetTabByType {
    template <typename Mailbox>
    static auto apply(const Change& in, const Mailbox& mailbox) {
        Delta out;
        out.value.reserve(in.changed.size());
        mailbox.tabs(extractTabTypes(in.changed), std::back_inserter(out.value));
        return out;
    }
};

template <>
struct ChangelogToDeltaImpl<changelog::FolderCreate> :
    GetFolderByFid<changelog::FolderCreate, delta::FolderCreate> {};

template <>
struct ChangelogToDeltaImpl<changelog::FolderModify> :
    GetFolderByFid<changelog::FolderModify, delta::FolderModify> {};

template <>
struct ChangelogToDeltaImpl<changelog::FolderDelete> {
    template <typename Mailbox>
    static auto apply(const changelog::FolderDelete& in, const Mailbox&) {
        delta::FolderDelete out;
        out.value.reserve(in.changed.size());
        boost::copy(extractFids(in.changed), std::back_inserter(out.value));
        return out;
    }
};

template <>
struct ChangelogToDeltaImpl<changelog::TabCreate> :
    GetTabByType<changelog::TabCreate, delta::TabCreate> {};

template <>
struct ChangelogToDeltaImpl<changelog::MoveToTab> {
    template <typename Mailbox>
    static auto apply(const changelog::MoveToTab& in, const Mailbox&) {
        delta::MoveToTab out;
        out.value.reserve(in.changed.size());
        boost::transform(in.changed, std::back_inserter(out.value), [&](auto& v) {
            delta::MoveToTab::Item item;
            item.mid = std::to_string(v.mid);
            item.tab = v.tab;
            return item;
        });
        return out;
    }
};

template <>
struct ChangelogToDeltaImpl<changelog::StickerCreate> {
    template <typename Mailbox>
    static auto apply(const changelog::StickerCreate& in, const Mailbox&) {
        delta::StickerCreate out;
        out.value.reserve(in.changed.size());
        boost::transform(in.changed, std::back_inserter(out.value), [&](auto& v) {
            delta::StickerCreate::Item item;
            item.mid = std::to_string(v.mid);
            item.stickerType = v.stickerType;
            return item;
        });
        return out;
    }
};

template <>
struct ChangelogToDeltaImpl<changelog::StickerRemove> {
    template <typename Mailbox>
    static auto apply(const changelog::StickerRemove& in, const Mailbox&) {
        delta::StickerRemove out;
        out.value.reserve(in.changed.size());
        boost::transform(in.changed, std::back_inserter(out.value), [&](auto& v) {
            delta::StickerRemove::Item item;
            item.mid = std::to_string(v.mid);
            item.stickerType = v.stickerType;
            return item;
        });
        return out;
    }
};

} // namespace hound::v2::changes
