#pragma once

#include <mail/hound/include/internal/v2/changes/change.h>
#include <mail/hound/include/internal/wmi/errors.h>
#include <yamail/expected.h> 

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes), Response,
    (std::vector<hound::v2::changes::Change>, changes)
)

BOOST_FUSION_DEFINE_STRUCT((hound::v2::changes), Request,
    (macs::Uid, uid)
    (macs::Revision, revision)
    (std::int64_t, max_count)
)

namespace hound::v2::changes {

namespace error = libwmi::error;
using mail_errors::error_code;

template <typename MailboxGetter>
struct Method {
    MailboxGetter getMailbox;

    yamail::expected<Response> operator() (Request args) const {
        if (args.max_count <= 0) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "max_count can not be less or equal to 0"});
        }

        if (args.uid.empty()) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "uid is empty"});
        }

        const auto mailbox = getMailbox(args.uid);

        std::vector<macs::Change> rchangelog;
        mailbox.getChanges(args.revision, args.max_count, std::back_inserter(rchangelog));

        if (rchangelog.empty()) {
            return Response{};
        }

        if (rchangelog.back().revision() != args.revision) {
            return yamail::make_unexpected(error_code{error::revisionNotFound, "revision has not been found in changelog"});
        }

        const auto changelog = boost::make_iterator_range(
                                    std::next(rchangelog.rbegin()), rchangelog.rend());
        Response out;
        composeChanges(mailbox, changelog, std::back_inserter(out.changes));
        return out;
    }
};

} // namespace hound::v2::changes
