#pragma once

#include <memory>
#include <boost/optional.hpp>

namespace hound::v2::detail {

template <typename T>
struct is_dereferencible : std::is_pointer<T> {};

template <typename T>
struct is_dereferencible<std::shared_ptr<T>> : std::true_type {};

template <typename T>
struct is_dereferencible<boost::shared_ptr<T>> : std::true_type {};

template <typename T, typename Dtor>
struct is_dereferencible<std::unique_ptr<T, Dtor>> : std::true_type {};

template <typename T>
struct is_dereferencible<boost::optional<T>> : std::true_type {};

template <typename T, typename = void>
struct deref_impl;

template <typename T>
struct deref_impl<T, std::enable_if_t<is_dereferencible<T>::value>> {
    template <typename Arg>
    constexpr static auto& apply(Arg&& v) { return *v; }
};

template <typename T>
struct deref_impl<T, std::enable_if_t<!is_dereferencible<T>::value>> {
    template <typename Arg>
    constexpr static auto& apply(Arg&& v) { return v; }
};

template <typename T>
constexpr decltype(auto) deref(T&& v) {
    return deref_impl<std::decay_t<T>>::apply(std::forward<T>(v));
}

} // namespace hound::v2::detail

