#pragma once

#include <macs/folder.h>
#include <yamail/data/reflection/details/adt.h>
#include <butil/datetime/date_utils.h>

namespace hound::server::handlers::v2::folders_tree {

struct Folder {
    Folder() = default;

    explicit Folder(const macs::Folder& f) :
        _folder(&f)
    {}

    Folder(const macs::Folder& f, const std::vector<Folder>& subfolders)
        : _folder(&f)
        , _subfolders(subfolders)
    {}

    const macs::Fid& id() const {
        return _folder->fid();
    }

    const macs::Fid& parentId() const {
        return _folder->parentId();
    }

    std::string displayName() const {
        return _folder->name();
    }

    bool isThreadable() const {
        return _folder->isThreadable();
    }

    bool isSubscribedForSharedFolder() const {
        return _folder->subscribedForSharedFolder();
    }

    bool isUnvisited() const {
        return _folder->unvisited();
    }

    bool isSystem() const {
        return _folder->isSystem();
    }

    const macs::Folder::Symbol symbolicName() const {
        return _folder->symbolicName();
    }

    std::size_t bytes() const {
        return _folder->bytes();
    }

    std::string type() const {
        return _folder->type().title();
    }

    std::string symbol() const {
        return _folder->symbolicName().title();
    }

    std::string createDate() const {
        const std::time_t t = DateUtils::string2timeFormat(_folder->creationTime(), "%s");
        std::tm tm = *std::localtime(&t);
        std::stringstream ss;
        ss << std::put_time(&tm, "%FT%T%z");
        return ss.str();
    }

    std::chrono::system_clock::time_point creationTime() const {
        return std::chrono::system_clock::from_time_t(DateUtils::string2timeFormat(_folder->creationTime(), "%s"));
    }

    std::size_t position() const {
        return _folder->position();
    }

    std::size_t unreadMessagesCount() const {
        return _folder->newMessagesCount();
    }

    std::size_t recentMessagesCount() const {
        return _folder->recentMessagesCount();
    }

    std::size_t messagesCount() const {
        return _folder->messagesCount();
    }

    std::int64_t revision() const {
        return _folder->revision();
    }

    const std::vector<Folder>& subfolders() const {
        return _subfolders;
    }

    const macs::Folder* _folder = nullptr;
    std::vector<Folder> _subfolders;
};

} // namespace hound::server::handlers::v2::folders_tree

#define RO(...) YREFLECTION_AUTO_PROP_RO(__VA_ARGS__)

YREFLECTION_ADAPT_ADT(hound::server::handlers::v2::folders_tree::Folder,
    RO(id)
    RO(name, displayName)
    RO(parentId)
    RO(type)
    RO(symbol)
    RO(createDate)
    RO(isThreadable)
    RO(isSubscribedForSharedFolder)
    RO(unreadMessagesCount)
    RO(recentMessagesCount)
    RO(messagesCount)
    RO(bytes)
    RO(revision)
    RO(isUnvisited)
    RO(subfolders)
)

#undef RO

