#pragma once

#include <mail/hound/include/internal/v2/folders_tree/folder_reflection.h>
#include <mail/hound/include/internal/v2/folders_tree/sorting.h>

#include <boost/optional.hpp>
#include <boost/locale.hpp>
#include <boost/fusion/adapted/struct/define_struct.hpp>
#include <user_journal/enumeration.h>
#include <macs/folder_set.h>
#include <mail/hound/include/internal/wmi/errors.h>
#include <yamail/expected.h>

BOOST_FUSION_DEFINE_STRUCT((hound::server::handlers::v2::folders_tree), Response,
    (std::vector<hound::server::handlers::v2::folders_tree::Folder>, folders_tree)
)

BOOST_FUSION_DEFINE_STRUCT((hound::server::handlers::v2::folders_tree), Request,
    (macs::Uid, uid)
    (std::string, sort)
    (boost::optional<std::string>, lang)
)

namespace hound::server::handlers::v2::folders_tree {

namespace error = libwmi::error;
using mail_errors::error_code;

inline yamail::expected<std::locale> lang2locale(const std::string& lang) {
    boost::locale::generator gen;
    if ("az" == lang) {
        return gen("az_AZ.UTF-8");
    } else if ("be" == lang) {
        return gen("be_BY.UTF-8");
    } else if ("en" == lang) {
        return gen("en_US.UTF-8");
    } else if ("hy" == lang) {
        return gen("hy_AM.UTF-8");
    } else if ("ka" == lang) {
        return gen("ka_GE.UTF-8");
    } else if ("kk" == lang) {
        return gen("kk_KZ.UTF-8");
    } else if ("ro" == lang) {
        return gen("ro_RO.UTF-8");
    } else if ("ru" == lang) {
        return gen("ru_RU.UTF-8");
    } else if ("tr" == lang) {
        return gen("tr_TR.UTF-8");
    } else if ("tt" == lang) {
        return gen("tt_RU.UTF-8");
    } else if ("uk" == lang) {
        return gen("uk_UA.UTF-8");
    } else {
        return yamail::make_unexpected(error_code{error::invalidArgument, "unsupported lang"});
    }
}

struct __SortingType {
    enum Enum {
        position,
        lang,
        unknown,
        date
    };

    using Map = user_journal::Enum2String<Enum>::Map;

    void fill(Map & map) const {
        map.insert(Map::value_type(position, "position"));
        map.insert(Map::value_type(lang, "lang"));
        map.insert(Map::value_type(date, "date"));
    }

    using Filler = __SortingType;
};

using SortingType = user_journal::Enumeration<__SortingType>;

template <typename MailboxGetter>
struct Method {
    MailboxGetter getMailbox;

    yamail::expected<Response> operator() (Request args) const {
        if (args.uid.empty()) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "uid is empty"});
        }
        if (args.sort.empty()) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "sorting type is empty"});
        }

        switch (SortingType::fromString(args.sort, std::nothrow)) {
            case SortingType::position: {
                const macs::FolderSet folderSet = getMailbox(args.uid).folders();
                boost::locale::generator gen;
                return Response{foldersSort(folderSet, PositionSortKey(), PathSortKey(gen("C")))};
            }
            case SortingType::lang: {
                if (!args.lang) {
                    return yamail::make_unexpected(error_code{error::invalidArgument, "argument lang must be passed for sorting type lang"});
                }

                const auto locale = lang2locale(*args.lang);
                if (!locale) {
                    return yamail::make_unexpected(locale.error());
                }

                const macs::FolderSet folderSet = getMailbox(args.uid).folders();
                return Response{foldersSort(folderSet, PathSortKey(locale.value()))};
            }
            case SortingType::date: {
                const macs::FolderSet folderSet = getMailbox(args.uid).folders();
                return Response{foldersSort(folderSet, LessByDateSortKey(), PathSortKey(lang2locale("ru").value()))};
            }
            case SortingType::unknown: {
                return yamail::make_unexpected(error_code{error::invalidArgument, "unknown sorting type " + args.sort});
            }
        }
    }
};

} // namespace hound::server::handlers::v2::folders_tree

