#pragma once

#include <mail/hound/include/internal/v2/folders_tree/folder_reflection.h>

#include <algorithm>
#include <unordered_map>
#include <iterator>
#include <cwctype>

#include <boost/algorithm/string/join.hpp>
#include <boost/algorithm/string/case_conv.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <boost/range/algorithm/sort.hpp>
#include <boost/range/algorithm/for_each.hpp>
#include <boost/range/adaptor/map.hpp>
#include <boost/locale/encoding_utf.hpp>

#include <macs/folder_set.h>

namespace hound::server::handlers::v2::folders_tree {

using ChildrenMap = std::unordered_multimap<macs::Fid, const macs::Folder*>;

inline ChildrenMap makeChildrenMap(const macs::FolderSet& folderSet){
    ChildrenMap out;

    boost::transform(folderSet, std::inserter(out, out.end()), [&] (auto& f) {
        return std::make_pair(f.second.parentId(), &f.second);
    });

    return out;
}

inline auto childrenOf(const ChildrenMap& map, const macs::Fid& parent) {
    auto [begin, end] = map.equal_range(parent);
    return boost::make_iterator_range(begin, end)
        | boost::adaptors::map_values;
};

template <typename Range, typename Less>
std::vector<Folder> makeFolders(const ChildrenMap& map, const Range& in, Less less) {
    std::vector<Folder> out;
    boost::transform(in, std::back_inserter(out), [&] (const macs::Folder* f) {
        return Folder(*f);
    });
    boost::sort(out, less);
    boost::for_each(out, [&map, &less] (Folder& f) {
         f._subfolders = makeFolders(map, childrenOf(map, f.id()), less);
    });
    return out;
}

template <typename Less>
std::vector<Folder> buildTree(const macs::FolderSet& folderSet, Less less) {
    const ChildrenMap map = makeChildrenMap(folderSet);
    return makeFolders(map, childrenOf(map, macs::Folder::noParent), less);
}

struct SymbolSortKey {
    using Symbol = macs::Folder::Symbol;

    const std::array<Symbol, 16> _positions{{
        Symbol::inbox,
        Symbol::none,
        Symbol::archive,
        Symbol::sent,
        Symbol::trash,
        Symbol::spam,
        Symbol::drafts,
        Symbol::template_,
        Symbol::outbox,
        Symbol::discount,
        Symbol::unsubscribe,
        Symbol::zombie_folder,
        Symbol::pending,
        Symbol::hidden_trash,
        Symbol::restored,
        Symbol::reply_later,
    }};

    static auto getSymbolPos(decltype(_positions)& positions, const Folder& f) {
        return std::distance(positions.begin(), std::find(positions.begin(), positions.end(), f.symbolicName()));
    };

    auto operator()(const Folder& f) const {
        return getSymbolPos(_positions, f);
    }
};

template <typename ... Keys>
struct Less {
    explicit Less(Keys ... keys)
        : _keys(std::make_tuple(SymbolSortKey(), std::forward<Keys>(keys)...))
    {}

    bool operator()(const Folder& lhs, const Folder& rhs) const {
        const auto lkeys = std::apply([&] (auto& ... keys) { return std::make_tuple(keys(lhs)...); }, _keys);
        const auto rkeys = std::apply([&] (auto& ... keys) { return std::make_tuple(keys(rhs)...); }, _keys);
        return lkeys < rkeys;
    }

    std::tuple<SymbolSortKey, Keys...> _keys;
};

struct PositionSortKey {
    auto operator()(const Folder& f) const {
        return f.position();
    }
};

struct LessByDateSortKey {
    auto operator()(const Folder& f) const {
        return f.creationTime();
    }
};

struct PathSortKey {
    explicit PathSortKey(std::locale locale) : _locale(locale) {}

    struct Key {
        const Folder& f;
        const std::locale& _locale;

        bool operator<(const Key& rhs) const {
            using namespace boost::locale::conv;
            std::wstring lhsName = utf_to_utf<wchar_t>(f.displayName());
            boost::algorithm::to_lower(lhsName);
            std::wstring rhsName = utf_to_utf<wchar_t>(rhs.f.displayName());
            boost::algorithm::to_lower(rhsName);
            return _locale(lhsName, rhsName);
        }
    };

    Key operator()(const Folder& f) const {
        return Key{f, _locale};
    }

    std::locale _locale;
};

template <typename ... Keys>
std::vector<Folder> foldersSort(const macs::FolderSet& folderSet, Keys ... keys) {
    return buildTree(folderSet, Less(std::forward<Keys>(keys)...));
}

} // namespace hound::server::handlers::v2::folders_tree

