#pragma once

#include <boost/fusion/adapted/struct/define_struct.hpp>
#include <mail/hound/include/internal/config.h>
#include <mail/hound/include/internal/wmi/errors.h>
#include <macs/folder.h>
#include <macs/label.h>
#include <macs/types.h>
#include <yamail/expected.h>

BOOST_FUSION_DEFINE_STRUCT((hound::server::handlers::v2::from_favorite_user_counter), Response,
    (uint64_t,  newMessagesCount)
)

BOOST_FUSION_DEFINE_STRUCT((hound::server::handlers::v2::from_favorite_user_counter), Request,
    (macs::Uid, uid)
    (size_t, counterLimit)
    (size_t, mailboxLimit)
)

namespace hound::server::handlers::v2::from_favorite_user_counter {

namespace error = libwmi::error;
using mail_errors::error_code;

template <typename MailboxGetter>
struct Method {
    MailboxGetter getMailbox;

    yamail::expected<Response> operator() (Request args) const {
        if (args.uid.empty()) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "uid is empty"});
        }

        const auto mailbox = getMailbox(args.uid);

        const macs::FolderSet folders = mailbox.folders();
        size_t countMessages = 0;
        for (const auto& f : folders) {
            countMessages += f.second.messagesCount();
        }
        if (countMessages > args.mailboxLimit) {
            return yamail::make_unexpected(error_code{error::tooManyMessages, "user has more then " + std::to_string(args.mailboxLimit) + " messages"});
        }

        const macs::LabelSet labels = mailbox.labels();
        const auto favoriteUserLabel = labels.find(macs::Label::Symbol::from_favorite_user_label);
        if (favoriteUserLabel == labels.end() || favoriteUserLabel->second.messagesCount() == 0)  {
            return Response{0};
        }

        const auto lid = favoriteUserLabel->second.lid();
        const auto countMessagesFromFavoriteUsers = mailbox.countNewMessagesWithLids(lid, args.counterLimit);

        return Response{countMessagesFromFavoriteUsers};
    }
};

} // namespace hound::server::handlers::v2::from_favorite_user_counter
