#pragma once

#include <mail/hound/include/internal/wmi/errors.h>

#include <macs/io.h>
#include <macs/types.h>

#include <boost/fusion/adapted/struct/define_struct.hpp>
#include <yamail/expected.h>

BOOST_FUSION_DEFINE_STRUCT((hound)(server)(handlers)(v2)(mids_by_tidsandlids), Response,
    (macs::Mids, mids)
)

BOOST_FUSION_DEFINE_STRUCT((hound)(server)(handlers)(v2)(mids_by_tidsandlids), Request,
    (macs::Tids, tids)
    (macs::Lids, lids)
)

namespace hound::server::handlers::v2::mids_by_tidsandlids {

namespace error = libwmi::error;

using macs::error_code;

template<typename MailboxGetter> struct Method {
    yamail::expected<Response> operator()(Request queryParams, std::size_t tidsLimit) const {
        if (queryParams.tids.empty()) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "TID parameters are required"});
        }

        if (queryParams.lids.empty()) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "LID parameters are required"});
        }

        const auto formatMessage([](const std::string& entity, uint32_t count, uint32_t limit) {
            std::stringstream message;
            message << entity << " count (" << count << ") is greater than the limit (" << limit << ")";
            return message.str();
        });

        if (queryParams.tids.size() > tidsLimit) {
            return yamail::make_unexpected(error_code{error::invalidArgument, formatMessage("TID", 
                    queryParams.tids.size(), tidsLimit)});
        }

        const auto emptyElementFound([](const auto& container) {
            return std::find(container.begin(), container.end(), typename std::remove_reference_t<decltype(
                    container)>::value_type{}) != container.end();
        });

        if (emptyElementFound(queryParams.tids)) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "TID must not be empty"});
        }

        if (emptyElementFound(queryParams.lids)) {
            return yamail::make_unexpected(error_code{error::invalidArgument, "LID must not be empty"});
        }

        const macs::Mids mids = getMailbox()
                .midsByTidsAndLids(std::move(queryParams.tids), std::move(queryParams.lids));

        return Response{mids};
    }

    MailboxGetter getMailbox;
};

}
