#pragma once

#include <boost/system/error_code.hpp>
#include <boost/system/system_error.hpp>

namespace libwmi::error {

enum Errors {
    ok                          =0,     /// No error
    unknown                     =1,     /// Unknown error
    internal                    =31,    /// internal error
    dbUnknownError              =1000,  /// Unknown DB error
    dbReadOnly                  =1001,  /// ORA-20103
    dbUniqueConstraint          =1003,  /// Unique constrain violated
    notInitialized              =1006,  /// User is not initialized

    noSuchMessage               =5000,  /// No such message
    invalidArgument             =5001,  /// Invalid argument
    noSuchFolder                =5002,  /// No such folder
    noSuchLabel                 =5005,  /// No such label
    folderNotEmpty              =5012,  /// Folder to be deleted is not empty
    revisionNotFound            =5013,  /// Given revision can not be found
    tooManyMessages             =5014,  /// User has too many messages
    noSuchTab                   =5021,  /// No such tab
    emptyResult                 =5022,

    uidNotFound                 =7001,  /// Uid not found
    regInProgress               =7002,  /// User registration in progress
};

}

namespace boost::system {

template <>
struct is_error_code_enum< ::libwmi::error::Errors> {
    static const bool value = true;
};

}

namespace libwmi::error {

using error_category = ::boost::system::error_category;
using error_code = ::boost::system::error_code;
using system_error = ::boost::system::system_error;

class Category : public error_category {
public:
    const char* name() const noexcept override { return "wmi"; }

    std::string message(int v) const override {
        switch(Errors(v)) {
            case ok :
                return "no error";
            case unknown :
                return "unknown error";
            case internal :
                return "internal error";
            case dbUnknownError :
                return "unknown database error";
            case dbReadOnly :
                return "ORA-20103 database is currently in read-only mode";
            case dbUniqueConstraint :
                return "unique constraint violated";
            case notInitialized :
                return "user is not initialized";
            case noSuchMessage :
                return "no such message";
            case invalidArgument :
                return "invalid argument";
            case noSuchFolder :
                return "no such folder";
            case noSuchLabel :
                return "no such label";
            case folderNotEmpty :
                return "folder to be deleted is not empty";
            case revisionNotFound :
                return "the revision can not be found";
            case tooManyMessages:
                return "user has too many messages";
            case uidNotFound:
                return "uid not found";
            case regInProgress:
                return "registration in progress";
            case noSuchTab:
                return "no such tab";
            case emptyResult:
                return "empty result from database";
        }
        return "unknown error " + std::to_string(v);
    }
};

inline const error_category& category() {
    static Category instance;
    return instance;
}

inline error_code make_error_code(Errors e) {
    return error_code(static_cast<int>(e), category());
}

}
