#pragma once

#include <macs/io.h>
#include <macs/folder.h>
#include <macs/queries/envelopes_sorting.h>
#include <macs_pg/service/service.h>

namespace hound {
namespace pins {

template <class Mailbox>
class BaseBuilder {
public:
    BaseBuilder(const Mailbox& mailbox, macs::Label label, size_t from, size_t count)
        : mailbox_(mailbox), label_(std::move(label)), from_(from), count_(count)
    {}

    virtual ~BaseBuilder() = default;

    BaseBuilder& sortType(macs::EnvelopesSorting sortType) {
        sortType_ = sortType;
        return *this;
    }

    template <typename Handler>
    auto get(Handler handler) const {
        macs::io::detail::init_async_result<Handler, macs::OnEnvelopeReceive> init(handler);
        byEntry(init.handler);
        return init.result.get();
    }

protected:
    virtual macs::EnvelopesQueryPtr getNonPinsQuery(size_t from, size_t count) const = 0;
    virtual macs::EnvelopesQueryPtr getPinsQuery(size_t from, size_t count) const = 0;
    // DARIA-49826 add query without pins filter
    virtual macs::EnvelopesQueryPtr getQueryWithoutPinsFilter(size_t from, size_t count) const = 0;
    virtual size_t getLabeledCount() const = 0;

    const Mailbox& mailbox_;
    macs::Label label_;
    macs::EnvelopesSorting sortType_;

private:
    void byEntry(macs::OnEnvelopeReceive handler) const {
        const auto labeledCount = getLabeledCount();

        if (labeledCount == 0) {
            auto wofQuery = getQueryWithoutPinsFilter(from_, count_);
            wofQuery->get(std::move(handler));
        } else if (from_ >= labeledCount) {
            auto nonPinsQuery = getNonPinsQuery(from_ - labeledCount, count_);
            nonPinsQuery->get(std::move(handler));
        } else if (from_ + count_ <= labeledCount) {
            auto pinsQuery = getPinsQuery(from_, count_);
            pinsQuery->get(std::move(handler));
        } else {
            auto pinsQuery = getPinsQuery(from_, labeledCount - from_);
            auto nonPinsQuery = getNonPinsQuery(0, count_ - labeledCount + from_);

            pinsQuery->get([handler = std::move(handler), nonPinsQuery = std::move(nonPinsQuery)]
                (macs::error_code e, auto envelope) mutable {
                    if(e || envelope) {
                        handler(std::move(e), std::move(envelope));
                    } else {
                        nonPinsQuery->get(std::move(handler));
                    }
            });
        }
    }

    size_t from_;
    size_t count_;
};

template <class Mailbox>
class FolderBuilder: public BaseBuilder<Mailbox> {
public:
    FolderBuilder(const Mailbox& mailbox, macs::Label label, size_t from, size_t count,
            macs::Folder folder)
        : BaseBuilder<Mailbox>(mailbox, std::move(label), from, count), folder_(std::move(folder))
    {}

private:
    size_t getLabeledCount() const {
        return this->label_.messagesCount();
    }

    macs::EnvelopesQueryPtr getNonPinsQuery(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto nonPinsQuery = builder.inFolder(folder_)
            .from(static_cast<int>(from))
            .count(count)
            .withoutLabel(this->label_)
            .sortBy(this->sortType_);
        return std::make_shared<macs::EnvelopesQueryInFolder>(nonPinsQuery);
    }

    macs::EnvelopesQueryPtr getPinsQuery(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto pinsQuery = builder.inMailbox()
            .from(static_cast<int>(from))
            .count(count)
            .withLabel(this->label_)
            .sortBy(this->sortType_);
        return std::make_shared<macs::EnvelopesQueryInMailbox>(pinsQuery);
    }

    macs::EnvelopesQueryPtr getQueryWithoutPinsFilter(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto wofQuery = builder.inFolder(folder_)
            .from(static_cast<int>(from))
            .count(count)
            .sortBy(this->sortType_);
        return std::make_shared<macs::EnvelopesQueryInFolder>(wofQuery);
    }

    macs::Folder folder_;
};

template <class Mailbox>
class ThreadBuilder: public BaseBuilder<Mailbox> {
public:
    ThreadBuilder(const Mailbox& mailbox, macs::Label label, size_t from, size_t count,
            std::string threadId)
        : BaseBuilder<Mailbox>(mailbox, std::move(label), from, count), threadId_(std::move(threadId))
    {}

private:
    size_t getLabeledCount() const {
        const auto threadLabelsList = this->mailbox_.threadLabels(macs::TidVector{threadId_});
        if (threadLabelsList.size() != 1) {
            throw std::runtime_error("error in ThreadPinsBuilder: getThreadLabels() returned "
                "strange number of items: " + std::to_string(threadLabelsList.size()));
        }
        const macs::ThreadLabels& threadLabels = threadLabelsList.front();
        for (const auto& label: threadLabels.labels()) {
            if (label.lid() == this->label_.lid()) {
                return label.cnt();
            }
        }
        return 0;
    }

    macs::EnvelopesQueryPtr getNonPinsQuery(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto nonPinsQuery = builder.inThread(threadId_)
            .from(static_cast<int>(from))
            .count(count)
            .withoutLabel(this->label_);
        return std::make_shared<macs::EnvelopesQueryInThread>(nonPinsQuery);
    }

    macs::EnvelopesQueryPtr getPinsQuery(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto pinsQuery = builder.inThread(threadId_)
            .from(static_cast<int>(from))
            .count(count)
            .withLabel(this->label_);
        return std::make_shared<macs::EnvelopesQueryInThread>(pinsQuery);
    }

    macs::EnvelopesQueryPtr getQueryWithoutPinsFilter(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto wofQuery = builder.inThread(threadId_)
            .from(static_cast<int>(from))
            .count(count);
        return std::make_shared<macs::EnvelopesQueryInThread>(wofQuery);
    }
    std::string threadId_;
};

template <class Mailbox>
class ThreadedBuilder: public BaseBuilder<Mailbox> {
public:
    ThreadedBuilder(const Mailbox& mailbox, macs::Label label, size_t from, size_t count,
            macs::Folder folder)
        : BaseBuilder<Mailbox>(mailbox, std::move(label), from, count), folder_(std::move(folder))
    {}

private:
    size_t getLabeledCount() const {
        return this->mailbox_.labelThreadsCount(this->label_.lid());
    }

    macs::EnvelopesQueryPtr getNonPinsQuery(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto nonPinsQuery = builder.inFolder(folder_)
            .from(static_cast<int>(from))
            .count(count)
            .withoutLabel(this->label_)
            .groupByThreads();
        return std::make_shared<macs::EnvelopesQueryInFolder>(nonPinsQuery);
    }

    macs::EnvelopesQueryPtr getPinsQuery(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto pinsQuery = builder.inMailbox()
            .from(static_cast<int>(from))
            .count(count)
            .withLabel(this->label_)
            .groupByThreads();
        return std::make_shared<macs::EnvelopesQueryInMailbox>(pinsQuery);
    }

    macs::EnvelopesQueryPtr getQueryWithoutPinsFilter(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto wofQuery = builder.inFolder(folder_)
            .from(static_cast<int>(from))
            .count(count)
            .groupByThreads();
        return std::make_shared<macs::EnvelopesQueryInFolder>(wofQuery);
    }

    macs::Folder folder_;
};

template <class Mailbox>
class TabBuilder: public BaseBuilder<Mailbox> {
public:
    TabBuilder(const Mailbox& mailbox, macs::Label label, size_t from, size_t count,
            macs::Tab tab)
        : BaseBuilder<Mailbox>(mailbox, std::move(label), from, count), tab_(std::move(tab))
    {}

private:
    size_t getLabeledCount() const {
        return this->label_.messagesCount();
    }

    macs::EnvelopesQueryPtr getNonPinsQuery(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto nonPinsQuery = builder.inTab(tab_.type())
            .from(static_cast<int>(from))
            .count(count)
            .withoutLabel(this->label_);
        return std::make_shared<macs::EnvelopesQueryInTab>(nonPinsQuery);
    }

    macs::EnvelopesQueryPtr getPinsQuery(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto pinsQuery = builder.inMailbox()
            .from(static_cast<int>(from))
            .count(count)
            .withLabel(this->label_);
        return std::make_shared<macs::EnvelopesQueryInMailbox>(pinsQuery);
    }

    macs::EnvelopesQueryPtr getQueryWithoutPinsFilter(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto wofQuery = builder.inTab(tab_.type())
            .from(static_cast<int>(from))
            .count(count);
        return std::make_shared<macs::EnvelopesQueryInTab>(wofQuery);
    }

    macs::Tab tab_;
};

template <class Mailbox>
class ThreadedTabBuilder: public BaseBuilder<Mailbox> {
public:
    ThreadedTabBuilder(const Mailbox& mailbox, macs::Label label, size_t from, size_t count,
            macs::Tab tab)
        : BaseBuilder<Mailbox>(mailbox, std::move(label), from, count), tab_(std::move(tab))
    {}

private:
    size_t getLabeledCount() const {
        return this->mailbox_.labelThreadsCount(this->label_.lid());
    }

    macs::EnvelopesQueryPtr getNonPinsQuery(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto nonPinsQuery = builder.inTab(tab_.type())
            .from(static_cast<int>(from))
            .count(count)
            .withoutLabel(this->label_)
            .groupByThreads();
        return std::make_shared<macs::EnvelopesQueryInTab>(nonPinsQuery);
    }

    macs::EnvelopesQueryPtr getPinsQuery(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto pinsQuery = builder.inMailbox()
            .from(static_cast<int>(from))
            .count(count)
            .withLabel(this->label_)
            .groupByThreads();
        return std::make_shared<macs::EnvelopesQueryInMailbox>(pinsQuery);
    }

    macs::EnvelopesQueryPtr getQueryWithoutPinsFilter(size_t from, size_t count) const {
        auto builder = this->mailbox_.query();
        auto wofQuery = builder.inTab(tab_.type())
            .from(static_cast<int>(from))
            .count(count)
            .groupByThreads();
        return std::make_shared<macs::EnvelopesQueryInTab>(wofQuery);
    }

    macs::Tab tab_;
};

template <class Mailbox>
class Factory {
public:
    Factory(Mailbox mailbox, macs::Label label, size_t from, size_t count)
        : mailbox_(std::move(mailbox)), label_(std::move(label)), from_(from), count_(count)
    {}

    ThreadedBuilder<Mailbox> threaded(macs::Folder folder) const {
        return ThreadedBuilder<Mailbox>(mailbox_, label_, from_, count_, std::move(folder));
    }

    FolderBuilder<Mailbox> folder(macs::Folder folder) const {
        return FolderBuilder<Mailbox>(mailbox_, label_, from_, count_, std::move(folder));
    }

    ThreadBuilder<Mailbox> threadId(std::string threadId) const {
        return ThreadBuilder<Mailbox>(mailbox_, label_, from_, count_, std::move(threadId));
    }

    TabBuilder<Mailbox> tab(macs::Tab tab) const {
        return TabBuilder<Mailbox>(mailbox_, label_, from_, count_, std::move(tab));
    }

    ThreadedTabBuilder<Mailbox> threadedTab(macs::Tab tab) const {
        return ThreadedTabBuilder<Mailbox>(mailbox_, label_, from_, count_, std::move(tab));
    }

private:
    Mailbox mailbox_;
    macs::Label label_;
    size_t from_;
    size_t count_;
};

} // namespace pins

namespace v1 {

template <typename Yield>
struct Mailbox {
    Mailbox(macs::Service& m, Yield yield) : metadata(m), yield(yield) {}

    template <class Range>
    auto threadLabels(Range&& tids) const {
        return metadata.threads().getThreadLabels(std::forward<Range>(tids), yield);
    }

    auto labelThreadsCount(const macs::Lid& lid) const {
        return metadata.labels().getThreadsCount(lid, yield);
    }

    auto query() const {
        return metadata.envelopes().query();
    }

    template <class Query>
    decltype(auto) fetch(Query&& query) const {
        return query.get(yield);
    }

private:
    macs::Service& metadata;
    Yield yield;
};

template <typename Yield>
class PinFactory {
public:
    PinFactory(macs::Service& metadata, Yield yield, macs::Label label, size_t from, size_t count)
        : factory(Mailbox<Yield>(metadata, yield), std::move(label), from, count)
    {}

    auto threaded(macs::Folder folder) const {
        return factory.threaded(std::move(folder));
    }

    auto folder(macs::Folder folder) const {
        return factory.folder(std::move(folder));
    }

    auto threadId(std::string threadId) const {
        return factory.threadId(std::move(threadId));
    }

private:
    pins::Factory<Mailbox<Yield>> factory;
};

} // namespace v1
} // namespace hound
