#include <mail/hound/include/internal/module.h>
#include <mail/hound/include/internal/server/handlers/ping.h>
#include <mail/hound/include/internal/server/handlers/attach_sid.h>
#include <mail/hound/include/internal/server/handlers/counters.h>
#include <mail/hound/include/internal/server/handlers/first_envelope_date.h>
#include <mail/hound/include/internal/server/handlers/folders.h>
#include <mail/hound/include/internal/server/handlers/folders_counters.h>
#include <mail/hound/include/internal/server/handlers/in_reply_to.h>
#include <mail/hound/include/internal/server/handlers/labels.h>
#include <mail/hound/include/internal/server/handlers/reset_fresh_counter.h>
#include <mail/hound/include/internal/server/handlers/reset_unvisited.h>
#include <mail/hound/include/internal/server/handlers/filter_search.h>
#include <mail/hound/include/internal/server/handlers/mailbox_revision.h>
#include <mail/hound/include/internal/server/handlers/yamail_status.h>
#include <mail/hound/include/internal/server/handlers/messages_by_folder.h>
#include <mail/hound/include/internal/server/handlers/messages_by_folder_without_label.h>
#include <mail/hound/include/internal/server/handlers/messages_by_label.h>
#include <mail/hound/include/internal/server/handlers/messages_by_thread.h>
#include <mail/hound/include/internal/server/handlers/messages_by_message_id.h>
#include <mail/hound/include/internal/server/handlers/messages_by_thread_with_pins.h>
#include <mail/hound/include/internal/server/handlers/messages_in_folder_with_pins.h>
#include <mail/hound/include/internal/server/handlers/messages_unread.h>
#include <mail/hound/include/internal/server/handlers/messages_unread_by_folder.h>
#include <mail/hound/include/internal/server/handlers/messages_with_attaches.h>
#include <mail/hound/include/internal/server/handlers/messages_unread_useful.h>
#include <mail/hound/include/internal/server/handlers/mimes.h>
#include <mail/hound/include/internal/server/handlers/nearest_messages.h>
#include <mail/hound/include/internal/server/handlers/threads_by_folder.h>
#include <mail/hound/include/internal/server/handlers/threads_info.h>
#include <mail/hound/include/internal/server/handlers/threads_in_folder_with_pins.h>
#include <mail/hound/include/internal/server/handlers/threads_participants.h>
#include <mail/hound/include/internal/server/handlers/imap_unsubscribed_folders.h>
#include <mail/hound/include/internal/server/handlers/deleted_messages.h>
#include <mail/hound/include/internal/v2/changes/v1_adaptor.h>
#include <mail/hound/include/internal/server/handlers/folder_tabs_new_counters.h>
#include <mail/hound/include/internal/wmi/yplatform/helpers/email.h>
#include <mail/hound/include/internal/v2/folders/v1_adaptor.h>
#include <mail/hound/include/internal/v2/folders_tree/v1_adaptor.h>
#include <mail/hound/include/internal/v2/mids_by_tids_and_lids/v1_adaptor.h>
#include <mail/hound/include/internal/v2/changelog/v1_adaptor.h>
#include <mail/hound/include/internal/v2/with_attaches_counters/v1_adaptor.h>
#include <mail/hound/include/internal/v2/from_favorite_user_counter/v1_adaptor.h>
#include <mail/hound/include/internal/v2/tabs/v1_adaptor.h>
#include <mail/hound/include/internal/v2/messages_by_tab/v1_adaptor.h>
#include <mail/hound/include/internal/v2/threads_by_tab/v1_adaptor.h>
#include <mail/hound/include/internal/v2/reset_tab_unvisited/v1_adaptor.h>
#include <mail/hound/include/internal/v2/messages_unread_by_tab/v1_adaptor.h>
#include <mail/hound/include/internal/v2/messages_in_tab_with_pins/v1_adaptor.h>
#include <mail/hound/include/internal/v2/threads_in_tab_with_pins/v1_adaptor.h>
#include <mail/hound/include/internal/v2/short_fresh_message/v1_adaptor.h>
#include <mail/hound/include/internal/v2/yamail_status/v1_adaptor.h>
#include <mail/hound/include/internal/v2/freezing_info/v1_adaptor.h>
#include <mail/hound/include/internal/v2/unfreeze_user/v1_adaptor.h>
#include <mail/hound/include/internal/v2/attach_sid/v1_adaptor.h>
#include <mail/hound/include/internal/v2/archive_change/v1_adaptor.h>
#include <mail/hound/include/internal/v2/archive_status/v1_adaptor.h>
#include <mail/hound/include/internal/v2/stickers/v1_adaptor.h>
#include <mail/hound/include/internal/v2/first_envelope_date/v1_adaptor.h>
#include <mail/hound/include/internal/v2/first_envelope_date_in_tab/v1_adaptor.h>


#include <mail/http_getter/client/include/module.h>

#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_httpclient/client.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>

#include <pa/async.h>
#include <stdexcept>


namespace hound {

template<class Handler>
void bindHandler(ConfigPtr config, ymod_webserver::server& server, std::shared_ptr<yplatform::reactor> reactor,
        const tvm_guard::Module& tvmGuard, const std::vector<std::string>& paths) {
    const auto handler = std::make_shared<Handler>(config);

    tvmGuard.bind(server, "", paths, [handler, reactor, config](ymod_webserver::response_ptr stream) {
        boost::asio::spawn(*(reactor->io()), [handler, stream](auto yield) {
            handler->process(stream, yield);
        }, boost::coroutines::attributes(config->coroutineStackSize()));
    });
}

template<class T>
void checkConfigExists(const std::string& name) {
    if (!yplatform::exists<T>(name)) {
        throw std::runtime_error("Cannot find config for module "+name);
    }
}

void Module::initPa(const yplatform::ptree& cfg) {
    pa::async_profiler::init(500000, 16, cfg.get<std::string>("profiler_log"));
}

void Module::init(const yplatform::ptree& cfg) {
    const std::string reactorName = cfg.get<std::string>("dependencies.reactor");
    const std::string serverName = cfg.get<std::string>("dependencies.server");
    const std::string httpGetterName = cfg.get<std::string>("dependencies.http_getter");
    const std::string tvmGuardName = cfg.get<std::string>("dependencies.tvm_guard");
    const std::string ymodTvmName = cfg.get<std::string>("dependencies.ymod_tvm");

    checkConfigExists<yplatform::reactor>(reactorName);
    checkConfigExists<ymod_webserver::server>(serverName);
    checkConfigExists<http_getter::TypedClientModule>(httpGetterName);
    checkConfigExists<tvm_guard::Module>(tvmGuardName);
    checkConfigExists<ymod_tvm::tvm2_module>(ymodTvmName);

    reactor = yplatform::find_reactor<std::shared_ptr>(reactorName);

    initPa(cfg);

    auto http = yplatform::find<http_getter::TypedClientModule, std::shared_ptr>(httpGetterName);

    config = makeConfig(cfg, std::move(http));

    const auto& recognizerCfg = config->recognizer();
    hound::helpers::recognizer(recognizerCfg.languageDict, recognizerCfg.languageWeights, recognizerCfg.encodingDict);

    bindHandlers(serverName, tvmGuardName);
    LOGDOG_(getModuleLogger(), notice, log::message="module initialized");
}

void Module::bindHandlers(const std::string& serverName, const std::string& tvmGuardName) {
    using namespace ::hound::server::handlers;

    auto server = yplatform::find<ymod_webserver::server, std::shared_ptr>(serverName);
    auto tvmGuard = yplatform::find<tvm_guard::Module, std::shared_ptr>(tvmGuardName);

    bindHandler<Ping>(config, *server, reactor, *tvmGuard, {"/ping"});
    bindHandler<AttachSid>(config, *server, reactor, *tvmGuard, {"/attach_sid"});
    bindHandler<CountersHandler>(config, *server, reactor, *tvmGuard, {"/counters"});
    bindHandler<FoldersHandler>(config, *server, reactor, *tvmGuard, {"/folders"});
    bindHandler<FoldersCountersHandler>(config, *server, reactor, *tvmGuard, {"/folders_counters"});
    bindHandler<FirstEnvelopeDate>(config, *server, reactor, *tvmGuard, {"/first_envelope_date"});
    bindHandler<FilterSearchHandler>(config, *server, reactor, *tvmGuard, {"/filter_search"});
    bindHandler<InReplyToHandler>(config, *server, reactor, *tvmGuard, {"/in_reply_to"});
    bindHandler<LabelsHandler>(config, *server, reactor, *tvmGuard, {"/labels"});
    bindHandler<ResetFreshCounter>(config, *server, reactor, *tvmGuard, {"/reset_fresh_counter"});
    bindHandler<ResetUnvisited>(config, *server, reactor, *tvmGuard, {"/reset_unvisited"});
    bindHandler<MailboxRevision>(config, *server, reactor, *tvmGuard, {"/mailbox_revision"});
    bindHandler<YamailStatus>(config, *server, reactor, *tvmGuard, {"/yamail_status"});
    bindHandler<MessagesByFolderHandler>(config, *server, reactor, *tvmGuard, {"/messages_by_folder"});
    bindHandler<MessagesByFolderWithoutLabelHandler>(config, *server, reactor, *tvmGuard, {"/messages_by_folder_without_label"});
    bindHandler<MessagesByLabelHandler>(config, *server, reactor, *tvmGuard, {"/messages_by_label"});
    bindHandler<MessagesByMessageIdHandler>(config, *server, reactor, *tvmGuard, {"/messages_by_message_id"});
    bindHandler<MessagesByThreadHandler>(config, *server, reactor, *tvmGuard, {"/messages_by_thread"});
    bindHandler<MessagesUnreadHandler>(config, *server, reactor, *tvmGuard, {"/messages_unread"});
    bindHandler<MessagesUnreadByFolderHandler>(config, *server, reactor, *tvmGuard, {"/messages_unread_by_folder"});
    bindHandler<MessagesUnreadUsefulHandler>(config, *server, reactor, *tvmGuard, {"/messages_unread_useful"});
    bindHandler<MessagesWithAttachesHandler>(config, *server, reactor, *tvmGuard, {"/messages_with_attaches"});
    bindHandler<MessagesByThreadWithPinsHandler>(config, *server, reactor, *tvmGuard, {"/messages_by_thread_with_pins"});
    bindHandler<MessagesInFolderWithPinsHandler>(config, *server, reactor, *tvmGuard, {"/messages_in_folder_with_pins"});
    bindHandler<Mimes>(config, *server, reactor, *tvmGuard, {"/mimes"});
    bindHandler<NearestMessagesHandler>(config, *server, reactor, *tvmGuard, {"/nearest_messages"});
    bindHandler<ThreadsByFolderHandler>(config, *server, reactor, *tvmGuard, {"/threads_by_folder"});
    bindHandler<ThreadsInfoHandler>(config, *server, reactor, *tvmGuard, {"/threads_info"});
    bindHandler<ThreadsInFolderWithPinsHandler>(config, *server, reactor, *tvmGuard, {"/threads_in_folder_with_pins"});
    bindHandler<ThreadsParticipantsHandler>(config, *server, reactor, *tvmGuard, {"/threads"});
    bindHandler<ImapUnsubscribedFolders>(config, *server, reactor, *tvmGuard, {"/imap_unsubscribed_folders"});
    bindHandler<DeletedMessagesHandler>(config, *server, reactor, *tvmGuard, {"/deleted_messages"});
    bindHandler<FolderTabsNewCountersHandler>(config, *server, reactor, *tvmGuard, {"/folder_tabs_new_counters"});

    bindHandler<hound::server::handlers::v2::archive::Handler>(config, *server, reactor, *tvmGuard, {"/v2/archive_change"});

    bindHandler<hound::server::handlers::v2::attach_sid::Handler>(config, *server, reactor, *tvmGuard, {"/v2/attach_sid"});

    bindHandler<hound::server::handlers::v2::changes::Handler>(config, *server, reactor, *tvmGuard, {"/v2/changes"});
    bindHandler<hound::server::handlers::v2::folders::Handler>(config, *server, reactor, *tvmGuard, {"/v2/folders"});
    bindHandler<hound::server::handlers::v2::folders_tree::Handler>(config, *server, reactor, *tvmGuard, {"/v2/folders_tree"});
    bindHandler<hound::server::handlers::v2::mids_by_tidsandlids::Handler>(config, *server, reactor, *tvmGuard, {"/v2/mids_by_tids_and_lids"});
    bindHandler<hound::server::handlers::v2::changelog::Handler>(config, *server, reactor, *tvmGuard, {"/v2/changelog"});
    bindHandler<hound::server::handlers::v2::with_attaches_counters::Handler>(config, *server, reactor, *tvmGuard, {"/v2/with_attaches_counters"});
    bindHandler<hound::server::handlers::v2::from_favorite_user_counter::Handler>(config, *server, reactor, *tvmGuard, {"/v2/from_favorite_user_counter"});

    bindHandler<hound::server::handlers::v2::tabs::Handler>(config, *server, reactor, *tvmGuard, {"/v2/tabs"});
    bindHandler<hound::server::handlers::v2::messages_by_tab::Handler>(config, *server, reactor, *tvmGuard, {"/messages_by_folder_and_tab"});
    bindHandler<hound::server::handlers::v2::messages_by_tab::Handler>(config, *server, reactor, *tvmGuard, {"/v2/messages_by_tab"});
    bindHandler<hound::server::handlers::v2::threads_by_tab::Handler>(config, *server, reactor, *tvmGuard, {"/threads_by_folder_and_tab"});
    bindHandler<hound::server::handlers::v2::threads_by_tab::Handler>(config, *server, reactor, *tvmGuard, {"/v2/threads_by_tab"});
    bindHandler<hound::server::handlers::v2::reset_tab_unvisited::Handler>(config, *server, reactor, *tvmGuard, {"/v2/reset_tab_unvisited"});
    bindHandler<hound::server::handlers::v2::messages_unread_by_tab::Handler>(config, *server, reactor, *tvmGuard, {"/v2/messages_unread_by_tab"});
    bindHandler<hound::server::handlers::v2::messages_in_tab_with_pins::Handler>(config, *server, reactor, *tvmGuard, {"/v2/messages_in_tab_with_pins"});
    bindHandler<hound::server::handlers::v2::threads_in_tab_with_pins::Handler>(config, *server, reactor, *tvmGuard, {"/v2/threads_in_tab_with_pins"});

    bindHandler<hound::server::handlers::v2::short_fresh_message::Handler>(config, *server, reactor, *tvmGuard, {"/v2/short_fresh_message"});
    bindHandler<hound::server::handlers::v2::yamail_status::Handler>(config, *server, reactor, *tvmGuard, {"/v2/yamail_status"});
    bindHandler<hound::server::handlers::v2::freezing_info::Handler>(config, *server, reactor, *tvmGuard, {"/v2/freezing_info"});
    bindHandler<hound::server::handlers::v2::unfreeze_user::Handler>(config, *server, reactor, *tvmGuard, {"/v2/unfreeze_user"});
    bindHandler<hound::server::handlers::v2::archive_status::Handler>(config, *server, reactor, *tvmGuard, {"/v2/archive_status"});
    bindHandler<hound::server::handlers::v2::first_envelope_date::Handler>(config, *server, reactor, *tvmGuard, {"/v2/first_envelope_date"});
    bindHandler<hound::server::handlers::v2::first_envelope_date_in_tab::Handler>(config, *server, reactor, *tvmGuard, {"/v2/first_envelope_date_in_tab"});

    bindHandler<hound::server::handlers::v2::stickers::Handler>(config, *server, reactor, *tvmGuard, {"/v2/stickers"});
}

} // namespace hound

DEFINE_SERVICE_OBJECT(hound::Module)
