#include <mail/hound/include/internal/server/context.h>
#include <mail/hound/include/internal/reflection/error.h>
#include <butil/split_key_value.h>
#include <boost/lexical_cast.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <yamail/data/serialization/json_writer.h>
#include <mail/hound/include/internal/wmi/common/error_codes.h>
#include <mail/hound/include/internal/wmi/yplatform/reflection/error.h>
#include <butil/network/real_ip_from_headers.h>
#include <boost/coroutine/exceptions.hpp>


namespace hound::server {

void makeReasonAndCodeFromCurrentException(std::string& reason, boost::system::error_code& code) {
    try {
        throw;
    } catch(const macs::UserNotInitialized& e) {
        code = libwmi::error::notInitialized;
        reason = e.what();
    } catch(const macs::ParamsException& e) {
        code = libwmi::error::invalidArgument;
        reason = e.what();
    } catch(const boost::system::system_error& e) {
        code = e.code();
        reason = e.what();
    } catch(const std::exception& e) {
        code = libwmi::error::unknown;
        reason = e.what();
    } catch (const boost::coroutines::detail::forced_unwind&) {
        throw;
    } catch(...) {
        code = libwmi::error::unknown;
        reason = "unhandled exception type";
    }
}

Context::Context(const ymod_webserver::http::stream_ptr& stream)
    : request(stream->request())
    , logger(getContextLogger(requestId(), boost::none))
    , method(stream->request()->url.make_full_path())
    , response_(stream)
{
    parseCookieHeader();
}

void Context::responseImpl(const std::string& body, ContentType contentType, ymod_webserver::codes::code code) {
    setCodeAndHeadersInResponse(code);

    switch (contentType) {
    case ContentType::textPlain:
        response_->set_content_type("text", "plain");
        break;
    case ContentType::applicationJson:
        response_->set_content_type("application", "json");
        break;
    }

    response_->result_body(body);
}

void Context::responsePong() {
    responseImpl("pong", ContentType::textPlain, ymod_webserver::codes::ok);
}

void Context::responseEmptyJson() {
    responseImpl("{}", ContentType::applicationJson, ymod_webserver::codes::ok);
}

void Context::responseInternalError(const boost::system::error_code& code, const std::string& reason) {
    responseErrorImpl(code, reason, ymod_webserver::codes::internal_server_error);
}

void Context::responseError(const boost::system::error_code& code,
                            const std::string& reason) {
    responseErrorImpl(code, reason, ymod_webserver::codes::ok);
}

void Context::responseError(const mail_errors::error_code& code) {
    responseErrorImpl(code.base(), code.what(), ymod_webserver::codes::ok);
}

void Context::responseErrorImpl(const boost::system::error_code& code,
                                const std::string& reason,
                                ymod_webserver::codes::code httpCode) {
    const auto wmiCode = errorCode2WmiError(code);
    LOGDOG_(logger, error, log::where_name=method, log::error_code=code,
            log::wmi_code=wmiCode.value(),
            log::message=reason);
    ErrorStruct res(wmiCode, reason);

    responseImpl(toJson(res, "error"), ContentType::applicationJson, httpCode);
}

void Context::responseErrorOnException() {
    boost::system::error_code code;
    std::string reason;

    makeReasonAndCodeFromCurrentException(reason, code);

    responseErrorImpl(code, reason, ymod_webserver::codes::ok);
}

void Context::responseBadRequestException() {
    boost::system::error_code code;
    std::string reason;

    makeReasonAndCodeFromCurrentException(reason, code);

    responseErrorImpl(code, reason, ymod_webserver::codes::bad_request);
}

void Context::responseInternalErrorException() {
    boost::system::error_code code;
    std::string reason;

    makeReasonAndCodeFromCurrentException(reason, code);

    responseErrorImpl(code, reason, ymod_webserver::codes::internal_server_error);
}

void Context::responseBadRequest(const boost::system::error_code& code, const std::string& reason) {
    responseErrorImpl(code, reason, ymod_webserver::codes::bad_request);
}

void Context::responseBadRequest(const mail_errors::error_code& code) {
    responseErrorImpl(code.base(), code.what(), ymod_webserver::codes::bad_request);
}

boost::optional<std::string> Context::getOptionalArg(const std::string& name) const {
    auto iter = request->url.params.find(name);
    return iter == request->url.params.end() ? boost::none : boost::make_optional(iter->second);
}

boost::optional<std::string> Context::getOptionalHeader(const std::string& name) const {
    const std::string n = boost::algorithm::to_lower_copy(name);
    auto iter = request->headers.find(n);

    return iter == request->headers.end() ? boost::none : boost::make_optional(iter->second);
}

std::string Context::getArg(const std::string& name) const {
    return getOptionalArg(name).get_value_or("");
}

bool Context::getArgument(const std::string& name, std::string& value) const {
    const auto i = request->url.params.find(name);
    if (i == request->url.params.end() || i->second.empty()) {
        return false;
    }
    value = i->second;
    return true;
}

bool Context::getBoolArg(const std::string& name) const {
    return getArg(name) == "yes";
}

void Context::setHeader(const std::string& name, const std::string& value) {
    headers.emplace_back(name, value);
}

std::string Context::getRawBody() const {
    return std::string(request->raw_body.begin(), request->raw_body.end());
}

std::string Context::getCookie(const std::string& name) const {
    const auto& val = cookies.find(name);
    if (cookies.end() == val) {
        return "";
    }

    return val->second;
}

void Context::parseCookieHeader() {
    const auto c = getOptionalHeader("cookie");
    if (c) {
        std::vector<std::string> cookiesList;
        boost::split(cookiesList, *c, boost::is_any_of(";"));
        boost::transform(cookiesList, std::inserter(cookies, cookies.end()),
                         boost::bind(splitKeyValueTrimmed, _1));
    }
}

void Context::setCodeAndHeadersInResponse(ymod_webserver::codes::code code) {
    response_->set_code(code);
    for (const auto& h : headers) {
        response_->add_header(h.first, h.second);
    }
}

std::string Context::testBuckets() const {
    return getOptionalHeader("X-Yandex-ExpBoxes").get_value_or("");
}

std::string Context::enabledTestBuckets() const {
    return getOptionalHeader("X-Yandex-EnabledExpBoxes").get_value_or("");
}

std::string Context::clientType() const {
    return getOptionalHeader("X-Yandex-ClientType").get_value_or("");
}

std::string Context::clientVersion() const {
    return getOptionalHeader("X-Yandex-ClientVersion").get_value_or("");
}

std::string Context::userAgent() const {
    return getOptionalHeader("User-Agent").get_value_or("");
}

std::string Context::requestId() const {
    return getOptionalHeader("X-Request-Id").get_value_or("");
}

std::string Context::getRealIp() const {
    return real_ip::getFromHeaders(getOptionalHeader("X-Forwarded-For"),
                                   getOptionalHeader("X-Real-Ip"),
                                   request->context->remote_address);
}

std::string Context::connectionId() const {
    return getOptionalArg("connection_id").get_value_or("");
}

void Context::setUid(const std::string& u) {
    uid = u;
    ::logdog::update_attribute(logger.get_context(), log::uid, u);
}

}
