#include <mail/hound/include/internal/wmi/service/subject/numcodes.h>
#include <mail/hound/include/internal/wmi/service/subject/rmrefwd.h>

#include <algorithm>
#include <cctype>
#include <cstring>

#include <sys/time.h>

typedef std::string::size_type Size;
const char* allOpsInString = "\02\03\04\05";

static const char         brack1[] = {'(', '{', '['};
static const char         brack2[] = {')', '}', ']'};

const std::string aReNames[] = {
      "Re",
      "HA",
      "\xCD\xE0", // \xCD\xE0: this must be windows1251 encoded
      "Vs",
};

const std::string aFwdNames[] = {
      "Fwd",
      "Fw",
};
const unsigned reCnt = sizeof(aReNames) / sizeof(*aReNames);
const unsigned fwdCnt = sizeof(aFwdNames) / sizeof(*aFwdNames);


enum TBlockType {
  T_BLOCK_TYPE_OTHER,
  T_BLOCK_TYPE_SUBJ_BLOB,
  T_BLOCK_TYPE_FWD_BLOCK
};


// functions find the longest substring starting from prepspecified position
static bool getLongestMatchingPrefix(const std::string & str, Size start,
        const std::string aNames[], Size cnt, std::string & sub)
{
    Size   maxSz = 0;
    std::string strLowerCase = str;
    for(Size i=0; strLowerCase[i]; i++) {
        strLowerCase[i] = static_cast<char>(tolower(strLowerCase[i]));
    }
    const char *   p1 = strLowerCase.c_str() + start;

    for (Size i = 0; i < cnt; ++i) {
        std::string aNameLowerCase = aNames[i];
        for(Size j=0; aNameLowerCase[j]; j++) {
            aNameLowerCase[j] = static_cast<char>(tolower(aNameLowerCase[j]));
        }
        const char *   p2 = aNameLowerCase.c_str();
        Size   sz = aNames[i].length();

        if (sz > maxSz && strncmp(p2, p1, sz) == 0) {
            maxSz = sz;
            sub = std::string(p1, sz);
        }
    }
    return maxSz != 0;
};
// function finds the longest substring ending in the prespecified position
static bool getLongestMatchingSuffix(const std::string & str, Size end,
                      const std::string aNames[], Size cnt, std::string & sub)
{
    Size   maxSz = 0;
    std::string strLowerCase = str;
    for(Size i=0; strLowerCase[i]; i++) {
        strLowerCase[i] = static_cast<char>(tolower(strLowerCase[i]));
    }
    const char *   p1 = strLowerCase.c_str() + end;

    for (Size i = 0; i < cnt; ++i) {
        std::string aNameLowerCase = aNames[i];
        for(Size j=0; aNameLowerCase[j]; j++) {
            aNameLowerCase[j] = static_cast<char>(tolower(aNameLowerCase[j]));
        }
        const char *   p2 = aNameLowerCase.c_str();
        Size   sz = aNames[i].length();

        if (sz > maxSz &&
            end + 1 >= sz &&
            strncmp(p2, p1 + 1 - sz, sz) == 0) {
            maxSz = sz;
            sub = std::string(p1 + 1 - sz, sz);
        }
    }
    return maxSz != 0;
};

static TBlockType detectBlockType(const std::string & str, Size start,
        Size & end, Size & endFwd, std::string & matchFwd)
{
    Size   strSz = str.length();

    if (start + 1 >= strSz || str[start] != '[') {
        return T_BLOCK_TYPE_OTHER;
    }

    Size i = start + 1;

    for ( ;i < str.length() && str[i] == ' '; ++i) { // empty body
    }

    if (i < strSz && str[strSz - 1] == ']') {
        if (getLongestMatchingPrefix(str, i, aFwdNames, fwdCnt, matchFwd)) {
            Size matchSz = matchFwd.length();
            if (i + matchSz < str.length() && str[i + matchSz] == ':') {
                endFwd = i + matchSz;
                end = strSz - 1;
                return T_BLOCK_TYPE_FWD_BLOCK;
            }
        }
    }

    for (Size i = start + 1; i < strSz; ++i) {
        if (str[i] == '[') {
            return T_BLOCK_TYPE_OTHER;
        }
        if (str[i] == ']') {
            end = i;
            return T_BLOCK_TYPE_SUBJ_BLOB;
        }
    }

    return T_BLOCK_TYPE_OTHER;
}
// e.g (8), [324] or {370}
bool detectBracketedNumber(const std::string & str,
        Size start, Size & end)
{
    if (start + 3 >= str.length()) { // at least three chars
        return false;
    }
    const char * p = std::find(brack1, brack1 + sizeof brack1, str[start]);

    if (p == brack1 + sizeof brack1) {
        return false;
    }
    ++start;
    if (!isdigit(str[start])) {
        return false;
    }
    for (;start < str.length() && isdigit(str[start]); ++start) {
    }
    if (start < str.length() && str[start] == brack2[p - brack1]) {
        end = start;
        return true;
    }
    return false;
}

void stripReFwd(const std::string & srcStr, std::string & resStr, std::string & sortOps, std::string & encOps, bool rmblobs)
{
// the stripping algo is based (NOT EXACTLY THE SAME) on the following document:
// IMAP SORT (internet draft)
// http://www.ietf.org/proceedings/01mar/I-D/imapext-sort-06.txt
    std::string str;
    // Rule 1: Convert all tabs and continuations to space.
    for(Size i = 0; i < srcStr.length();) {
        const char c = srcStr[i];
        if ((c >= 0 && c < 32) || isspace(c)) {
            str += ' ';
            ++i;
        } else {
            str += srcStr[i++];
        }
    }
    bool cont;
    // We need to count how much digits we need to encode Re() numbers

    const unsigned int numSizeRe = 2;  /* this covers 62 * 62 = 3844 answeres, which is more than enough
     for correct sorting of re(<some_num>) */
    encOps = "";
    sortOps = "";

    // General note about stripping-loops we finish loop only when no stripping
    // was done on the current iteration
    do {
        cont = false;
        Size sz;
        // Rule 2: Remove all trailing text of the subject that matches
        // "(" <FORWARD_KEYWORD> ")" | <SPACE>
        // repeat until no more matches are possible.
        bool removed;
        do {
            removed = false;
            sz = str.length();
            if (!sz)
                break;
            if (str[sz - 1] == ' ') {
                Size b = sz - 1;
                for (;;--b) {
                    if (!b || str[b - 1] != ' ')
                        break;
                }

                encOps.append(std::string(1, suffixOpChar) + std::string(sz - b, ' '));
                str.erase(b, sz - b);
                removed = true;
            } else if (str[sz - 1] == ')' && sz > 2) {
                std::string matchFwd;
                if (getLongestMatchingSuffix(str, sz - 2, aFwdNames, fwdCnt, matchFwd)) {
                    int          start = static_cast<int>(sz - 2 - matchFwd.length());
                    Size rmSize = matchFwd.length() + 2;

                    if (start >= 0 && str[static_cast<size_t>(start)] == '(') {
                        std::string s(1, suffixOpChar);

                        if (start >= 1 && str[static_cast<size_t>(start - 1)] == ' ') {
                            while (start >= 1 && str[static_cast<size_t>(start - 1)] == ' ') {
                                --start;
                                ++rmSize;
                                s += ' ';
                            }
                        }
                        s += '(';
                        s += matchFwd;
                        s += ')'; // here we add only lowercase fwd
                        encOps.append(s);
                        sortOps.insert(sortOps.begin(), 1, 'F'); // forward

                        str.erase(static_cast<size_t>(start), rmSize);
                        removed = true;
                    }
                }
            }
            cont |= removed;
        } while (removed);
        /* Rule 3:  Remove all prefix text of the subject that matches
           <SPACE>+ <SUBJ_BLOB>* | <SUBJ_RE> ":" | <SUBJ_FWD> ":"
           <SUBJ_BLOB> ::= "[" <NON_SQUARE_BRACKET_CHAR>* "]"
           # The only exception of SUBJ_BLOB!!!!:
           "[" <SPACE>* <FWD_KEYWORD> ":" <NON_SQUARE_BRACKET_CHAR>* "]"
           for example [acm128] is subj_blob, but [ fwd: acm128] is not!
           <SUBJ_RE> ::= <RE_KEYWORD> <SPACE>* <BRACKETED_NUMBER>* <SPACE>* <SUBJ_BLOB>+ ":"
           <SUBJ_FWD> ::= <FWD_KEYWORD> <SPACE>* <SUBJ_BLOB>+ ":" |
           Rule 3 modification: there could be also constructions:
           (fwd)
        */
        do {
            std::string matchFwd;

            removed = false;
            sz = str.length();
            if (!sz)
                break;
            if (str[0] == ' ') {
                Size e = 0;
                while (e < sz && str[e] == ' ') {
                    ++e;
                }
                str.erase(0, e);
                encOps.append(std::string(1, prefixOpChar) + std::string(e, ' '));
                removed = true;
            } else if (str[0] == '(' &&
                       getLongestMatchingPrefix(str, 1, aFwdNames, fwdCnt, matchFwd) &&
                       str.length() > 1 + matchFwd.length() &&
                       str[1 + matchFwd.length()] == ')') {
                Size e = 2 + matchFwd.length() ;

                while (e < str.length() && str[e] == ' ') {
                    ++e;
                }
                std::string s = std::string(1, prefixOpChar) + str.substr(0, e);

                encOps.append(s);
                sortOps.insert(sortOps.begin(), 1, 'F'); // forward
                str.erase(0, e);
                removed = true;
            } else if (str[0] == '[') {
                std::string   matchFwd;
                Size  e, e1;
                TBlockType tp = detectBlockType(str, 0, e, e1, matchFwd);
                if (tp == T_BLOCK_TYPE_SUBJ_BLOB && rmblobs) {
                    std::string s(1, prefixOpChar);

                    s += str.substr(0, e + 1);
                    while (e + 1 < str.length() && str[e + 1] == ' ') {
                        s += ' ';
                        ++e;
                    }
                    encOps.append(s);
                    str.erase(0, e + 1);
                    removed = true;
                } else if (tp == T_BLOCK_TYPE_FWD_BLOCK) {
                    std::string s(1, bracketOpChar);

                    s += "[";
                    encOps.append(s);

                    s = std::string(1, prefixOpChar);
                    if (e1 - matchFwd.length() - 1) { // there are some spaces between '[' and fwd
                        s += std::string(e1 - matchFwd.length() - 1, ' ');
                    }

                    s += matchFwd;
                    s += ":";
                    encOps.append(s);
                    sortOps.insert(sortOps.begin(), 1, 'F'); // forward

                    str.erase(e, 1);
                    str.erase(0, e1 + 1);

                    removed = true;
                }
            } else {
                // let's find the longest possible Re keyword
                std::string   matchReFwd;
                int           f = 0;
                Size  e = 0;
                if (getLongestMatchingPrefix(str, 0, aReNames, reCnt, matchReFwd)) {
                    e = matchReFwd.length();
                    f = 1;
                } else if (getLongestMatchingPrefix(str, 0, aFwdNames, fwdCnt, matchReFwd)) {
                    f = 2;
                    e = matchReFwd.length();
                }

                if (f) {
                    Size e2 = e;
                    int          reNum = 0;

                    if (f == 1) { // re can have optional number but fwd cannot
                        Size tmpe;
                        if (detectBracketedNumber(str, e, tmpe)) {
                            reNum = atoi(str.substr(e + 1, tmpe - e).c_str());
                            e2 = tmpe + 1;
                        }
                    }

                    while (e2 < str.length()) {
                        Size  tmp, e3;
                        std::string   tmps;

                        if (str[e2] == ' ') {
                            ++e2;
                        } else if (rmblobs && detectBlockType(str, e2, e3, tmp, tmps)
                                                       == T_BLOCK_TYPE_SUBJ_BLOB) {
                            e2 = e3 + 1;
                        } else {
                            break;
                        }
                    };
                    if (e2 < str.length() && str[e2] == ':') {
                        std::string tmps(1, prefixOpChar);

                        while (e2 + 1 < str.length() && str[e2 + 1] == ' ') {
                            ++e2;
                        }
                        tmps += str.substr(0, e2 + 1);
                        encOps.append(tmps);

                        if (f == 1) {
                            std::string   tmps("R");
                            tmps += encNum62(static_cast<unsigned int>(reNum), numSizeRe, true); // reply with a number, 0 by default

                            sortOps.insert(0, tmps);
                        } else {
                            sortOps.insert(sortOps.begin(), 1, 'F');
                        }
                        str.erase(0, e2 + 1);
                        removed = true;
                    }
                }
            }
            cont |= removed;
        } while (removed);
    } while (cont);

    resStr = str;
}


int reconstrPartsReFwd(const std::string & strippedStr, std::string& prefix,
                       std::string & resStr, std::string& suffix,
                       const std::string& encOps)
{

    resStr = strippedStr;

    if (!encOps.length()) {
        return 0;
    }

    Size ppos = encOps.length() - 1;
    do {
        const char * p = allOpsInString;
        Size pos = ppos,
                     sz = strlen(p);

        for(;;--pos) {
            if (std::find(p, p + sz, encOps[pos]) != p + sz) {
              break;
            }
            if (!pos) { // incorrect options no command
                return -4;
            }
        }

        if (ppos - pos < 1) { // there is not command less than 2 chars
            return -1;
        }
        std::string opt = encOps.substr(pos, ppos - pos + 1);
        ppos = (pos) ? pos - 1: 0;

        if (opt[0] == bracketOpChar) {
            const char * p = std::find(brack1, brack1 + sizeof brack1, opt[1]);
            if (p == brack1 + sizeof brack1) { // invalid bracket
                return -2;
            }
            prefix = std::string(1, *p) + prefix;
            suffix += brack2[p - brack1];
        } else if (opt[0] == prefixOpChar ||
                   opt[0] == suffixOpChar) {
            bool   doPref = (opt[0] == prefixOpChar);
            std::string s = opt.substr(1, opt.length() - 1);

            if (doPref) {
                prefix = s + prefix;
            } else {
                suffix += s;
            }
        } else {
            return -3;
        }
    } while (ppos);
    return 0;
}
