#include <mail/hound/include/internal/v2/archive_change/v1_adaptor.h>
#include <mail/hound/include/internal/wmi/service/passport.h>
#include <mail/hound/include/internal/user_state.h>

#include <yamail/data/serialization/json_writer.h>

#include <boost/lexical_cast.hpp>
#include <yplatform/find.h>
#include <mail/http_getter/client/include/module.h>
#include <mail/logdog/include/logdog/logger.h>
#include <macs/archive_error.h>


namespace {

inline macs::ArchiveState convertState(const std::string& s) {
    macs::ArchiveState archiveState;
    if (!yamail::data::reflection::from_string<macs::ArchiveState>(s, archiveState)) {
        throw macs::ParamsException("Unknown archive state: " + s);
    }
    return archiveState;
}

}


namespace hound::server::handlers::v2::archive {

void Handler::execute(RequestPtr r, boost::asio::yield_context yield) const {
    auto& request = *r;
    try {
        executeMacs(r, getMetadata(request), yield);
    } catch (const macs::ParamsException&) {
        request.responseBadRequestException();
    } catch (const boost::system::system_error& e) {
        const auto& ec = e.code();
        bool isBadRequest = ec == sharpei::client::Errors::UidNotFound || ec == sharpei::client::Errors::HttpCode;
        if (!isBadRequest && ec.category() == macs::getExchangeArchiveStateCategory()) {
            using EnumType = macs::ExchangeArchiveStateStatus;
            switch(EnumType(ec.value())) {
                case EnumType::user_not_here: break;
                case EnumType::user_wrong_state:
                case EnumType::archive_not_found:
                    isBadRequest = true;
                    break;
            }
        }
        if (isBadRequest) {
            request.responseBadRequestException();
        } else {
            request.responseInternalErrorException();
        }
    } catch (...) {
        request.responseInternalErrorException();
    }
}

void Handler::executeMacs(RequestPtr request, MailMetadataPtr metadata, boost::asio::yield_context yield) const {
    auto yy = macs::io::make_yield_context(yield);

    const auto uid = request->getArg("uid");
    const auto requestId = request->requestId();

    const auto action = request->getArg("action");
    const auto intermediateState = convertState(fmt::format("{}_requested", action));
    const auto finalState = convertState(fmt::format("{}_in_progress", action));

    metadata->users().exchangeArchiveState(
        macs::UserState::archived,
        macs::ArchiveState::archivation_complete, intermediateState,
        std::nullopt, yy
    );

    const auto httpClient = config().httpClient()->create(*(request->request), nullptr);
    auto passportResponse = passportUnfreeze(httpClient, config().passportEndpoint(), uid, yield);

    if (!passportResponse.hasError()) {
        metadata->users().activateUserAndUpdateArchiveState(intermediateState, finalState, yy);
        internal::logUserStateChange(boost::lexical_cast<uint64_t>(uid), requestId, "archived");
        return request->responseEmptyJson();
    }

    if (passportResponse.hasOnlyRetriableErrors()) {
        request->responseInternalError(
            libwmi::error::internal,
            "retriable error from passport: " + yamail::data::serialization::toJson(passportResponse).str()
        );
    } else {
        request->responseBadRequest(
            libwmi::error::invalidArgument,
            "non-retriable error from passport: " + yamail::data::serialization::toJson(passportResponse).str()
        );
    }
}

}
