#include <mail/hound/include/internal/server/handlers/filter_search.h>
#include <mail/hound/include/internal/wmi/meta_access/metadata_helpers.h>
#include <mail/hound/include/internal/wmi/yplatform/filter_arguments.h>
#include <mail/hound/include/internal/wmi/yplatform/filter_search_params.h>
#include <mail/hound/include/internal/wmi/yplatform/reflection/envelopes_with_mailbox_revision.h>

#include <boost/range/algorithm/copy.hpp>
#include <boost/range/adaptor/map.hpp>
#include <boost/algorithm/string/join.hpp>

namespace hound::server::handlers {

using namespace macs;

bool FilterSearchHandler::getFolderSymbolsParam(Request& request,
        const std::string& paramName, std::list<Folder::Symbol> &symbols) const {
    std::list<string> symbolNames;
    request.getArgList(paramName, std::back_inserter(symbolNames));

    for (const auto &name : symbolNames) {
        const auto& symb = Folder::Symbol::getByTitle(name);
        if (symb.title() != name) {
            setErrorHeader(request);
            request.responseError(libwmi::error::invalidArgument,
                    "bad parameter " + paramName + ": unknown folder symbol " + name);
            return false;
        }
        symbols.push_back(symb);
    }
    return true;
}

bool FilterSearchHandler::getFoldersFilter(Request& request, FoldersFilter &filter) const {
    if (!getFolderSymbolsParam(request, "incl_folders", filter.includeFolders)
            || !getFolderSymbolsParam(request, "excl_folders", filter.excludeFolders)) {
        return false;
    }

    filter.onlyUseful = request.getOptionalArg("only_useful").is_initialized();

    return true;
}

bool FilterSearchHandler::checkFoldersFilter(Request& request, FoldersFilter &filter) const {
    if (filter.onlyUseful && !filter.includeFolders.empty()) {
        setErrorHeader(request);
        request.responseError(libwmi::error::invalidArgument,
                "only_useful and incl_folders parameters can not be used together");
        return false;
    }


    if (filter.onlyUseful && !filter.excludeFolders.empty()) {
        setErrorHeader(request);
        request.responseError(libwmi::error::invalidArgument,
                "only_useful and excl_folders parameters can not be used together");
        return false;
    }

    if (!filter.includeFolders.empty() && !filter.excludeFolders.empty()) {
        setErrorHeader(request);
        request.responseError(libwmi::error::invalidArgument,
                "excl_folders and incl_folders parameters can not be used together");
        return false;
    }

    return true;
}

void FilterSearchHandler::filterFids(FoldersFilter &filter, macs::Service& metadata,
        std::list<string> &fids, boost::asio::yield_context yieldCtx) const {
    auto yield = macs::io::make_yield_context(yieldCtx);
    if (filter.onlyUseful) {
        const macs::FolderSet& folders = metadata.folders().getAllFolders(yield);
        helpers::filterUsefulFolders(folders, fids);
    }

    if (!filter.includeFolders.empty()) {
        const macs::FolderSet& folders = metadata.folders().getAllFoldersWithHidden(yield);
        helpers::filterFoldersWithSymbols(folders, filter.includeFolders, fids);
    }

    if (!filter.excludeFolders.empty()) {
        const macs::FolderSet& folders = metadata.folders().getAllFolders(yield);
        helpers::filterFoldersWithoutSymbols(folders, filter.excludeFolders, fids);
    }
}

bool FilterSearchHandler::getFids(Request& request, macs::Service& metadata,
        std::list<string> &fids, boost::asio::yield_context yieldCtx) const {
    request.getArgList("fids", std::back_inserter(fids));
    filterEmptyItems(fids, request.logger, "fids", "filter_search");

    FoldersFilter filter;
    if (!getFoldersFilter(request, filter) || !checkFoldersFilter(request, filter)) {
        return false;
    }

    if (filter.onlyUseful || !filter.includeFolders.empty() || !filter.excludeFolders.empty()) {
        if (fids.empty()) {
            auto yield = macs::io::make_yield_context(yieldCtx);
            if (!filter.includeFolders.empty()) {
                const auto& folders = metadata.folders().getAllFoldersWithHidden(yield);
                boost::copy(folders | boost::adaptors::map_keys, std::back_inserter(fids));
            } else {
                const auto& folders = metadata.folders().getAllFolders(yield);
                boost::copy(folders | boost::adaptors::map_keys, std::back_inserter(fids));
            }
        }

        filterFids(filter, metadata, fids, yieldCtx);

        if (fids.empty()) {
            request.response(std::vector<helpers::Envelope>{}, "envelopes");
            return false;
        }
    }

    return true;
}

bool FilterSearchHandler::getMids(Request& request, std::list<string> &mids) const {
    request.getArgList("mids", std::back_inserter(mids));
    filterEmptyItems(mids, request.logger, "mids", "filter_search");

    if (mids.empty()) {
        const auto& methodName = request.method;
        LOGDOG_(request.logger, notice, log::where_name=methodName,
                log::message="mids are not specified");
        request.response(std::vector<helpers::Envelope>{}, "envelopes");
        return false;
    }

    return true;
}

bool FilterSearchHandler::getLids(Request& request, std::list<string> &lids) const {
    request.getArgList("lids", std::back_inserter(lids));
    filterEmptyItems(lids, request.logger, "lids", "filter_search");

    return true;
}

bool FilterSearchHandler::getFilterSearchParams(Request& request, macs::Service& metadata,
        FilterSearchParams &params, boost::asio::yield_context yield) const {
    if (!getFids(request, metadata, params.fids, yield) ||
            !getMids(request, params.mids) || !getLids(request, params.lids)) {
        return false;
    }

    params.onlyUnread = request.getOptionalArg("unread").is_initialized();
    params.onlyAttachments = request.getOptionalArg("only_attachments").is_initialized();

    params.folderSet = filter_search::getFolderSetQueryPart(request.getArg("folder_set"));
    params.order = filter_search::getEnvelopesSorting(request.getArg("order"));
    params.isFullFoldersAndLabels = getLogicalParam(request, "full_folders_and_labels");
    params.needMailboxRevision = getLogicalParam(request, "need_mailbox_revision");

    return true;
}

bool FilterSearchHandler::getLogicalParam(Request& request, const std::string& paramName) {
    std::string value;
    if (request.getArgument(paramName, value)) {
        return filter_search::toBoolean(value, paramName);
    } else {
        return false;
    }
}

template <typename T>
void FilterSearchHandler::response(Request & request,
        boost::optional<macs::Revision> mailboxRevision, T&& envelopes) const {
    if (envelopes.size() == 0) {
        request.setHeader("X-Yandex-Hound-Filter-Search-Empty", "yes");
    }
    if (mailboxRevision) {
        const auto result = helpers::envelopesWithRevision(std::forward<T>(envelopes), *mailboxRevision);
        request.response(result, "filter_search");
    } else {
        request.response(envelopes, "envelopes");
   }
}

void FilterSearchHandler::setErrorHeader(Request& request) const {
    request.setHeader("X-Yandex-Hound-Filter-Search-Error", "yes");
}

void FilterSearchHandler::executeMacs(RequestPtr request, MailMetadataPtr metadata, boost::asio::yield_context yieldCtx) const {
    FilterSearchParams params;

    if (!getFilterSearchParams(*request, *metadata, params, yieldCtx)) {
        return;
    }

    auto yield = macs::io::make_yield_context(yieldCtx);
    auto mailboxRevision = boost::optional<macs::Revision>(boost::none);
    if (params.needMailboxRevision) {
        mailboxRevision = metadata->folders().getMailboxRevision(yield);
    }

    const auto result = metadata->envelopes().query()
            .filterSearch()
            .onlyUnread(params.onlyUnread)
            .onlyAtta(params.onlyAttachments)
            .fids(params.fids)
            .lids(params.lids)
            .mids(params.mids)
            .folderSet(params.folderSet)
            .sortBy(params.order)
            .get(yield);

    std::vector<helpers::Envelope> envelopeList{std::begin(result), std::end(result)};
    const std::string& methodName = request->method;

    if (envelopeList.empty()) {
        LOGDOG_(request->logger, warning, log::where_name = methodName, log::caller = request->getOptionalArg("caller"),
                log::error_code = make_error_code(libwmi::error::emptyResult));
    }

    if (params.isFullFoldersAndLabels) {
        const auto &folders = metadata->folders().getAllFoldersWithHidden(yield);
        const auto &labels = metadata->labels().getAllLabels(yield);
        auto envelopes = helpers::detailedEnvelopes(std::move(envelopeList), folders, labels);

        using boost::algorithm::join;

        for (const auto &x : envelopes.missingFids) {
            LOGDOG_(request->logger, error, log::where_name = methodName, log::fid = x.first, log::mids = x.second,
                    log::message = "folder not found");
        }

        for (const auto &x : envelopes.missingLids) {
            LOGDOG_(request->logger, error, log::where_name = methodName, log::lid = x.first, log::mids = x.second,
                    log::message = "label not found");
        }

        response(*request, mailboxRevision, std::move(envelopes.values));
    } else {
        response(*request, mailboxRevision, envelopeList);
    }
}

}
