#include <mail/hound/include/internal/server/handlers/macs.h>
#include <mail/hound/include/internal/wmi/meta_access/logging.h>

namespace hound::server::handlers {

MacsExecuter::MacsExecuter(UserStrategy userStrategy)
    : userStrategy(userStrategy) {}

std::vector<UserType> MacsExecuter::getUserTypes() const {
    switch (userStrategy) {
        case UserStrategy::existingOnly: return {UserType::existing};
        case UserStrategy::existingThenDeleted: return {UserType::existing, UserType::deleted};
    }
}

MacsHandler::MailMetadataPtr MacsHandler::getMetadata(Request& request, UserType userType) const {
    const std::string uid = getUserId(request);
    request.setUid(uid);

    boost::optional<std::string> serviceDbUser = getServiceDbUser(request);
    auto logger = createMacsPgLogger(request.logger);
    auto serviceParams = ymod_maildb::ServiceParams {
        .uid=uid,
        .realIp=request.getRealIp(),
        .requestId=request.requestId(),
        .module="hound",
        .userType=userType
    };
    if (serviceDbUser) {
        serviceParams.credentials = macs::pg::Credentials {
            .user=std::move(serviceDbUser.value()), 
            .password=""
        };
    }

    auto journalParams = ymod_maildb::UserJournalParams {
        .connectionId=request.connectionId(),
        .expBoxes=request.testBuckets(),
        .enabledExpBoxes=request.enabledTestBuckets(),
        .clientType=request.clientType(),
        .clientVersion=request.clientVersion(),
        .userAgent=request.userAgent(),
        .yandexUid=request.getCookie("yandexuid"),
        .iCookie=request.getCookie("i"),
        .sessionInfo=request.getArg("session_info"),
    };
    return config().mailDb()->service(std::move(serviceParams), std::move(journalParams), std::move(logger), getQueryStrategy(request));
}

std::string MacsHandler::getUserId(const Request& request) const {
    const auto& uid = request.getOptionalArg("uid");

    if (!uid || uid->empty()) {
        throw macs::ParamsException("uid parameter is required");
    }

    return uid.get();
}

::macs::pg::QueryHandleStrategy MacsHandler::getQueryStrategy(const Request& request) const {
    const auto& dbtype = request.getOptionalArg("dbtype");

    if (!dbtype) {
        return defaultQueryStrategy();
    }

    if ("master" == *dbtype) {
        return ::macs::pg::readMasterThenReplica;
    } else if ("replica" == *dbtype) {
        return ::macs::pg::readReplicaThenMaster;
    }

    throw macs::ParamsException("dbtype parameter should be master or replica");
}

boost::optional<std::string> MacsHandler::getServiceDbUser(const Request& request) const {
    const auto& serviceName = request.getOptionalArg("caller");
    if (!serviceName || serviceName->empty()) {
        return boost::none;
    }

    const auto service = config().servicesDbUsers().find(*serviceName);
    if (config().servicesDbUsers().end() != service) {
        return service->second;
    } else {
        const auto methodName = __PRETTY_FUNCTION__;
        LOGDOG_(request.logger, warning, log::where_name=methodName, log::message="unknown caller " + *serviceName);
    }
    return boost::none;
}

}
