#pragma once

#include <mail/unistat/cpp/include/meters/common.h>

#include <mail/hound/include/internal/wmi/errors.h>

#include <boost/accumulators/accumulators.hpp>
#include <boost/accumulators/statistics.hpp>
#include <boost/range/adaptors.hpp>
#include <re2/re2.h>

namespace unistat {

struct XdbErrorsTop {
    explicit XdbErrorsTop(std::size_t topNumber, std::string namePrefix)
        : _topNumber(topNumber)
        , _regex("host=([^\\s]+)")
        , _name(normalizeName(namePrefix))
        , _counters()
    {}

    void update(std::string_view record) {
        std::string xdbHost;
        const re2::StringPiece recordPiece(record.data(), record.size());
        if (re2::RE2::PartialMatch(recordPiece, _regex, &xdbHost)) {
            _counters[xdbHost] += 1;
        }
    }

    void update(const std::map<std::string, std::string>& record) {
        auto it = record.find("connection_info");
        if (record.end() != it) {
            std::string xdbHost;
            const re2::StringPiece recordPiece(it->second);
            if (re2::RE2::PartialMatch(recordPiece, _regex, &xdbHost)) {
                _counters[xdbHost] += 1;
            }
        }
    }

    std::vector<NamedValue<std::size_t>> get() const {
        using namespace boost::accumulators;

        accumulator_set<std::pair<std::size_t, std::string>, stats<tag::tail<right>>> acc(tag::tail<right>::cache_size = _topNumber);
        for (const auto& p : _counters) {
            acc(std::make_pair(p.second, p.first));
        }

        std::vector<NamedValue<std::size_t>> result;
        result.reserve(_topNumber);
        std::transform(tail(acc).begin(), tail(acc).end(), std::back_inserter(result), [this] (const auto& p) {
            return NamedValue<std::size_t>{withSigoptSuffix(_name + "_" + p.second, "summ"), p.first};
        });
        return result;
    }

private:
    const std::size_t _topNumber;
    const re2::RE2 _regex;
    const std::string _name;
    std::unordered_map<std::string, std::size_t> _counters;
};

struct PggFallbackMeter {
    explicit PggFallbackMeter(std::string namePrefix)
        : _namePrefix(normalizeName(namePrefix))
        , _counters()
    {}

    void update(const std::map<std::string, std::string>& record) {
        auto itRuleName = record.find("rule_name");
        auto itStrategyName = record.find("strategy_name");

        if (record.end() != itRuleName && record.end() != itStrategyName) {
            std::string name = itStrategyName->second + "_in_" + itRuleName->second;
            _counters[name] += 1;
        }
    }

    std::vector<NamedValue<std::size_t>> get() const {
        std::vector<NamedValue<std::size_t>> result;
        result.reserve(_counters.size());
        std::transform(_counters.begin(), _counters.end(), std::back_inserter(result), [this] (const auto& p) {
            return NamedValue<std::size_t>{normalizeName(withSigoptSuffix(_namePrefix + "_" + p.first, "summ")), p.second};
        });
        return result;
    }

private:
    std::string _namePrefix;
    std::unordered_map<std::string, std::size_t> _counters;
};

struct EmptyResultFilterSearch {
    explicit EmptyResultFilterSearch(std::string namePrefix)
        : _namePrefix(normalizeName(namePrefix))
        , _counter()
    {}

    void update(const std::map<std::string, std::string>& record) {
        const auto whereName = record.find("where_name");
        const auto ecValue = record.find("error_code.value");

        if (record.end() != whereName && "filter_search" == whereName->second
                && record.end() != ecValue && "5022" == ecValue->second) {
            _counter += 1;
        }
    }

    NamedValue<std::size_t> get() const {
        const auto name = normalizeName(withSigoptSuffix(_namePrefix + "filter_search_empty_result", "summ"));
        return NamedValue<std::size_t>{name, _counter};
    }

private:
    std::string _namePrefix;
    std::size_t _counter;
};

struct CountWmiErrors {
    explicit CountWmiErrors(std::vector<int> expectedErrors, std::string namePrefix)
        : _namePrefix(std::move(namePrefix))
        , _unexpected(0)
        , _errors()
    {
        _unexpected = 0;
        for (auto& e: expectedErrors) {
            _errors[e] = 0;
        }
    }

    void update(const std::map<std::string, std::string>& record) {
        const auto level = record.find("level");
        if (record.end() == level || "error" != level->second) {
            return;
        }

        const auto ecValue = record.find("wmi_code");
        if (record.end() != ecValue) {
            const int code = std::stoi(ecValue->second);
            if (_errors.end() == _errors.find(code)) {
                _unexpected += 1;
            } else {
                _errors[code] += 1;
            }
        }
    }

    std::vector<NamedValue<std::size_t>> get() const {
        const auto metrics = _errors | boost::adaptors::transformed([&prefix = _namePrefix](const auto& e){
            const std::string name = normalizeName(buildSignalName(prefix, libwmi::error::category().message(e.first)));
            return NamedValue<std::size_t>{withSigoptSuffix(name, "summ"), e.second};
        });
        std::vector<NamedValue<std::size_t>> result {metrics.begin(), metrics.end()};

        const std::string unexpectedName = normalizeName(buildSignalName(_namePrefix, "unexpected_error"));
        result.emplace_back(withSigoptSuffix(unexpectedName, "summ"), _unexpected);

        return result;
    }

private:
    std::string _namePrefix;
    std::size_t _unexpected;
    std::map<int, std::size_t> _errors;
};

} // namespace unistat
