from argparse import ArgumentParser
from collections import namedtuple
from mail.unistat.cpp.cython.meters import (
    AccessLogCount,
    AccessLogCountByFirstStatusDigit,
    AccessLogCountByPathAndFirstStatusDigit,
    AccessLogRequestTimeHist,
    AccessLogRequestTimeHistByPath,
    CountErrors,
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
    PaLogRequestTimeHist,
    SupervisorLogRestartMeters,
)
from mail.hound.unistat.cpp.run import (
    XdbErrorsTop,
    PggFallbackMeter,
    EmptyResultFilterSearch,
    CountWmiErrors,
)
import os
import yaml
import logging
import mail.hound.unistat.cpp.run as hound_unistat
import mail.unistat.cpp.cython.logs as logs
import re


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


HOUND_ENDPOINTS = [
    '/attach_sid?',
    '/filter_search?',
    '/first_envelope_date?',
    '/folder_tabs_new_counters?',
    '/messages_by_folder_and_tab?',
    '/threads_by_folder_and_tab?',
    '/v2/changes?',
    '/v2/folders?',
    '/v2/folders_tree?',
]


HOUND_LOG_ERRORS = [
    "sharpei_client error",
    "user is not initialized in the database",
    "empty result from database",
    "canceling statement due to lock timeout",
    "Not implemented",
]

HOUND_WMI_ERRORS = [
    1,      # Unknown error
    31,     # Internal error
    1000,   # Unknown DB error
    1003,   # Unique constrain violated
    1006,   # User is not initialized
    5000,   # No such message
    5001,   # Invalid argument
    5002,   # No such folder
    5005,   # No such label
    5013,   # Given revision can not be found
    5014,   # User has too many messages
    5021,   # No such tab
    5022,   # Empty result from DB
    7001,   # Uid not found
    7002,   # User registration in progress
]


def make_access_log_meters(name_prefix, endpoints):
    meters = [
        AccessLogCount(name_prefix),
        AccessLogCountByFirstStatusDigit(name_prefix),
        AccessLogRequestTimeHist((0, 20, 100, 300, 1000), 'access_log_request'),
    ]

    for endpoint in endpoints:
        meters += [
            AccessLogCountByPathAndFirstStatusDigit(endpoint, name_prefix),
            AccessLogRequestTimeHistByPath((0, 20, 100, 300, 1000), endpoint, 'access_log_request'),
        ]

    return meters


def make_supervisor_meters():
    return [
        SupervisorLogRestartMeters('supervisor')
    ]


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='/app/log/unistat.log', help='path for yplatform.log')
    parser.add_argument('--supervisorlog', help='path for supervisord.log')
    parser.add_argument('hound_config')
    return parser.parse_args()


def make_hound_log_meters(errors):
    meters = []

    for err in errors:
        meters += [
            CountErrors(err)
        ]
    topNumber = 5
    meters += [
        CountWmiErrors(HOUND_WMI_ERRORS, "wmi_errors"),
        XdbErrorsTop(topNumber, "xdb_errors"),
        PggFallbackMeter("fallback"),
        EmptyResultFilterSearch("")
    ]

    return meters


def get_module_config(data, name):
    return next(v['configuration'] for v in data['config']['modules']['module'] if v['system']['name'] == name)


def get_logger_sink_path(data, name, sink_index=0):
    return os.path.join(os.curdir, data['config']['log'][name]['sinks'][sink_index]['path'])

def make_hound_config(data):
    remove_scheme = re.compile(r"https?://")
    sharpei_host = get_module_config(data, 'sharpei_client')['hosts']
    sharpei_host = remove_scheme.sub('', sharpei_host).strip()
    return HoundConfig(
        httpclient_log=get_logger_sink_path(data, 'http_client'),
        access_log=get_logger_sink_path(data, 'access'),
        application_log=get_logger_sink_path(data, 'hound'),
        profiler_log=os.path.join(os.curdir, get_module_config(data, 'hound')['profiler_log']),
        user_journal_log=get_logger_sink_path(data, 'user_journal'),
        stat_server='localhost: %s ' % get_module_config(data, 'stat_server')['endpoints']['listen']['_port'],
        sharpei_host=sharpei_host,
    )


def make_hound_http_client_log_meters():
    return [
        HttpClientHttpRequestCountByStatus("conninfo", "count_by_status_sharpei"),
        HttpClientHttpRequestTotalTimeHist((0, 20, 100, 300, 1000), "conninfo", "sharpei_time_hist"),
    ]


def make_hound_pa_log_meters():
    return [
        PaLogRequestTimeHist((0, 10, 20, 30, 40, 50, 100, 500, 1000, 5000), "postgresql", "pa"),
    ]

HoundConfig = namedtuple('HoundConfig', (
    'httpclient_log',
    'access_log',
    'application_log',
    'profiler_log',
    'user_journal_log',
    'stat_server',
    'sharpei_host',
))


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.hound_config) as f:
        hound_config = make_hound_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    access_log = logs.AccessTskv([], make_access_log_meters("access_log", HOUND_ENDPOINTS), fast_forward, hound_config.access_log)
    hound_log = hound_unistat.HoundLog([], make_hound_log_meters(HOUND_LOG_ERRORS), fast_forward, hound_config.application_log)
    http_client_log = logs.HttpClientLog([], make_hound_http_client_log_meters(), fast_forward, hound_config.httpclient_log)
    pa_log = logs.PaLog([], make_hound_pa_log_meters(), fast_forward, hound_config.profiler_log)

    logs_list = [hound_log, access_log, http_client_log, pa_log]

    if args.supervisorlog is not None:
        supervisor_log = logs.SupervisorLog(
            [],
            make_supervisor_meters(),
            fast_forward,
            args.supervisorlog
        )
        logs_list.append(supervisor_log)

    hound_unistat.run(args.host, args.port, logs_list, args.log, logLevel='info')


if __name__ == '__main__':
    main()
