#include <mail/http_getter/client/include/endpoint.h>
#include <mail/http_getter/client/include/exception.h>
#include <mail/http_getter/client/include/endpoint_reflection.h>
#include <yamail/data/serialization/json_writer.h>


namespace http_getter {

EndpointException::EndpointException(const std::string& msg, const Endpoint& e)
    : std::runtime_error(fmt::format("{}, endpoint={}", msg, e))
{ }

std::ostream& operator<<(std::ostream& out, const Endpoint& e) {
    if (e.empty()) {
        out << "{}";
    } else {
        out << yamail::data::serialization::JsonWriter<Endpoint::Data>(e.data()).result();
    }

    return out;
}

bool Endpoint::empty() const {
    return !data_.has_value();
}

constexpr char SLASH = '/';

void Endpoint::dataConstraint() const {
    const Data& d = data();

    if (d.fallback && d.tries < 2) {
        throw EndpointException("number of tries must be two or higher", *this);
    }

    if (!d.method.empty() && d.method.front() != SLASH) {
        throw EndpointException("method should start from '/' symbol", *this);
    }

    if (d.url.empty()) {
        throw EndpointException("url must not be empty", *this);
    }

    if (d.fallback && d.fallback->empty()) {
        throw EndpointException("fallback must not be empty", *this);
    }

    if (d.tries == 0) {
        throw EndpointException("tries must not be zero", *this);
    }
}

void Endpoint::fixDoubleSlash() {
    Data& d = data();
    if (d.url.back() == SLASH) {
        d.url.pop_back();
    }
    if (d.fallback && d.fallback->back() == SLASH) {
        d.fallback->pop_back();
    }
}

void Endpoint::checkDataIsNotEmpty() const {
    if (empty()) {
        throw EndpointException("Uninitialized endpoint", *this);
    }
}

Endpoint::Endpoint(Endpoint::Data&& raw)
    : data_(std::move(raw))
{
    checkAndFixData();
}

void Endpoint::setData(Endpoint::Data&& d) {
    data_ = std::move(d);
    checkAndFixData();
}

void Endpoint::setData(const Endpoint::Data& d) {
    data_ = d;
    checkAndFixData();
}

const Endpoint::Data& Endpoint::data() const {
    checkDataIsNotEmpty();
    return *data_;
}

Endpoint::Data& Endpoint::data() {
    checkDataIsNotEmpty();
    return *data_;
}

void Endpoint::checkAndFixData() {
    dataConstraint();
    fixDoubleSlash();
}

}
