#include <iostream>

#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <yamail/data/deserialization/ptree.h>

#include <tvm_guard/module.h>

#include <logdog/backend/yplatform_log.h>
#include <logdog/attributes/mail_attributes.h>

#include <mail/http_getter/client/include/module.h>
#include <mail/http_getter/client/include/endpoint_reflection.h>

namespace example {

YREFLECTION_DEFINE_ENUM_INLINE(Type,
    endpoint,
    background
)

constexpr static auto formatter = ::logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-example-tskv-log"));

inline auto getModuleLogger() {
    return ::logdog::make_log(
        formatter,
        std::make_shared<yplatform::log::source>(YGLOBAL_LOG_SERVICE, "example")
    );
}

using ModuleLogger = decltype(getModuleLogger());

constexpr static auto RETRY_BODY = "please retry";
struct Mocks: public yplatform::module {
    void init(const yplatform::ptree&) {
        auto server = yplatform::find<ymod_webserver::server>("webserver");
        auto guard = yplatform::find<tvm_guard::Module>("tvm_guard");

        guard->bind(*server, "", {"/endpoint"}, [](ymod_webserver::response_ptr s) {
            s->result(ymod_webserver::codes::ok, "I");
        });

        guard->bind(*server, "", {"/123/fallback"}, [](ymod_webserver::response_ptr s) {
            s->result(ymod_webserver::codes::ok, " love ");
        });

        guard->bind(*server, "", {"/unstable"}, [](ymod_webserver::response_ptr s) {
            static int strangeNonatomicCounterWithoutMutex = 0;

            switch (strangeNonatomicCounterWithoutMutex++) {
                case 0: s->result(ymod_webserver::codes::internal_server_error, "Syktyvkar"); break;
                case 1: s->result(ymod_webserver::codes::internal_server_error, "Kyakhta"); break;
                case 2: s->result(ymod_webserver::codes::ok, RETRY_BODY); break;
                case 3: s->result(ymod_webserver::codes::ok, RETRY_BODY); break;
                default:
                    s->result(ymod_webserver::codes::ok, "Verkhoyansk");
                    strangeNonatomicCounterWithoutMutex = 0; break;
            }
        });

        guard->bind(*server, "", {"/background"}, [](ymod_webserver::response_ptr s) {
            s->result(
                ymod_webserver::codes::ok,
                std::string(s->request()->raw_body.begin(), s->request()->raw_body.end())
            );
        });
    }
};

struct Server: public yplatform::module {
    void init(const yplatform::ptree& cfg) {
        auto server = yplatform::find<ymod_webserver::server>("webserver");
        auto getter = yplatform::find<http_getter::ClientModule>("http_getter");
        auto typedGetter = yplatform::find<http_getter::TypedClientModule>("typed_http_getter");

        auto reactor = yplatform::find_reactor<std::shared_ptr>("global");

        using namespace yamail::data::deserialization;
        auto endpoint = fromPtree<http_getter::Endpoint>(cfg.get_child("endpoint"));
        auto fallback = fromPtree<http_getter::Endpoint>(cfg.get_child("fallback"));
        auto unstable = fromPtree<http_getter::TypedEndpoint>(cfg.get_child("unstable"));

        http_getter::Endpoint background({
            .url="http://localhost:14488",
            .method="/background",
            .tvm_service="mocks_tvm_service",
            .keep_alive=false,
            .log_response_body=true,
            .log_post_body=true,
            .timeout_ms={
                .connect=std::chrono::milliseconds{200},
                .total=std::chrono::milliseconds{500}
            },
            .tries=6
        });

        auto handler = [=] (boost::asio::yield_context ctx, ymod_webserver::response_ptr s) {
            using namespace http_getter::detail::operators;

            std::string uid, reqId;
            if (auto iter = s->request()->url.params.find("uid"); iter != s->request()->url.params.end()) {
                uid = iter->second;
            }

            if (auto iter = s->request()->headers.find("x-request-id"); iter != s->request()->headers.end()) {
                reqId = iter->second;
            }

            auto requestStats = http_getter::withLog(getter->httpLogger(uid, reqId));
            auto client = getter->create(*s->request(), requestStats);
            auto typed = typedGetter->create(*s->request(), requestStats);
            auto yield = io_result::make_yield_context(ctx);


            std::string result;
            client
                ->req(client->toGET(endpoint).getArgs("uid"_arg=uid))
                ->call(Type::endpoint, http_getter::withDefaultHttpWrap([&](yhttp::response resp) {
                    result += resp.body;
                }), yield);


            client
                ->req(client->toGET(
                        fallback.format(fmt::arg("uid", uid))
                    )
                )
                ->call("fallback", http_getter::Handler {
                    .error=[](boost::system::error_code ec) {
                        LOGDOG_(getModuleLogger(), notice, logdog::message=ec.message());
                    },
                    .success=[&](yhttp::response resp) {
                        result += resp.body;
                        return http_getter::Result::success;
                    }
                }, yield);


            typed
                ->req(typed->toGET(unstable))
                ->call("unstable", [&](yhttp::response resp) {
                        if (resp.body == RETRY_BODY) {
                            return http_getter::Result::retry;
                        }
                        result += resp.body;
                        return http_getter::Result::success;
                    }, yield);


            client
                ->req(client->toPOST(background).body(uid))
                ->backgroundCall(Type::background, http_getter::withDefaultHttpWrap([](yhttp::response resp) {
                    LOGDOG_(getModuleLogger(), notice, logdog::message="background call, body is: " + resp.body);
                }));


            s->result(ymod_webserver::codes::ok, result);
        };

        server->bind("", {"/handler"}, [=] (ymod_webserver::response_ptr stream) {
            boost::asio::spawn(*reactor->io(), [=](boost::asio::yield_context yield) {
                handler(yield, stream);
            }, boost::coroutines::attributes(1048576));
        });
    }
};

}

DEFINE_SERVICE_OBJECT(example::Server)
DEFINE_SERVICE_OBJECT(example::Mocks)


int main(int argc, char* argv[]) {
    if (argc != 2) {
        std::cout << "usage " << argv[0] << " <config>\n";
        return 1;
    }

    return yplatform_start(argv[1]);
}
