#pragma once

#include <mail/http_getter/client/include/typed_client.h>

#include <mail/http_getter/client/include/tvm.h>
#include <mail/http_getter/client/include/endpoint.h>
#include <mail/http_getter/client/include/helpers.h>
#include <mail/http_getter/client/include/request_status.h>
#include <mail/http_getter/client/include/metrics.h>
#include <mail/http_getter/client/include/logger.h>

#include <butil/http/headers.h>
#include <butil/http/arguments.h>
#include <http_getter/http_request.h>
#include <io_result/io_result.h>


namespace http_getter {

template<class Builder>
struct RequestLoop: public std::enable_shared_from_this<RequestLoop<Builder>> {
    RequestStatsPtr lai_;
    Builder builder_;
    AsyncRun run_;

    template<class Context>
    void makeAsyncCall(const std::string& operation, unsigned i, Handler handler,
                       Context ctx) {
        auto self = this->shared_from_this();
        const auto start = std::chrono::steady_clock::now();
        Request req = (i > 0 && builder_.hasFallback())
                ? builder_.fallback()
                : builder_.primary();

        run_(std::move(req),
            [self, start, operation, handler, i, ctx, options = req.options]
            (const boost::system::error_code& ec, yhttp::response resp) mutable {
                const auto milliseconds = std::chrono::duration_cast<std::chrono::milliseconds>(
                    std::chrono::steady_clock::now() - start
                );
                const double seconds = static_cast<double>(milliseconds.count()) / 1000.;
                Result retry = Result::retry;
                std::optional<std::string> bodyToLog;

                if (ec) {
                    handler.error(ec);
                    if (self->lai_) {
                        self->lai_->responseIsGot(operation, i, seconds, bodyToLog, ec);
                    }
                } else {
                    if (self->lai_) {
                        bodyToLog = options.logResponseBody.value_or(false)
                                ? resp.body
                                : makeHiddenValue(resp.body);
                    }

                    try {
                        retry = handler.success(std::move(resp));

                        if (self->lai_) {
                            self->lai_->responseIsGot(
                                operation, i, seconds, *bodyToLog,
                                static_cast<unsigned>(resp.status), retry
                            );
                        }
                    } catch (const boost::coroutines::detail::forced_unwind&) {
                        throw;
                    } catch (const boost::system::system_error& ex) {
                        handler.error(ex.code());
                        if (self->lai_) {
                            self->lai_->responseIsGot(operation, i, seconds, bodyToLog, ex.code());
                        }
                    } catch (const std::exception& ex) {
                        handler.error(
                            boost::system::error_code(
                                static_cast<int>(Errors::UnexpectedException),
                                getErrorCategory()
                            )
                        );
                        if (self->lai_) {
                            self->lai_->responseIsGot(operation, i, seconds, bodyToLog, ex);
                        }
                    }
                }

                if (retry == Result::retry) {
                    const unsigned nextI = i + 1;
                    if (nextI >= options.maxAttempts.value_or(0)) {
                        ctx(mail_errors::error_code());
                        if (self->lai_) {
                            self->lai_->finishing(operation, retry);
                        }
                    } else {
                        self->makeAsyncCall(operation, nextI, handler, ctx);
                    }
                } else {
                    ctx(mail_errors::error_code());
                    if (self->lai_) {
                        self->lai_->finishing(operation, retry);
                    }
                }
            }
        );
    }

    template<class Operation, class Context>
    auto makeAsyncCall(Operation operation, Handler handler, Context ctx) {
        io_result::detail::init_async_result<Context, io_result::Hook<void>> init{ctx};

        std::string operationName;

        if (lai_) {
            operationName = getOperationName(operation);
            lai_->starting(operationName);
        }

        makeAsyncCall(operationName, 0, std::move(handler), init.handler);

        return init.result.get();
    }

public:

    RequestLoop(Builder builder, AsyncRun run, RequestStatsPtr lai)
        : lai_(std::move(lai))
        , builder_(std::move(builder))
        , run_(std::move(run))
    { }

    template<class H, class Operation>
    void backgroundCall(Operation operation, H handler) {
        makeAsyncCall(operation, adapted(std::move(handler)), io_result::use_future);
    }

    template<class Operation, class H, class Context = io_result::sync_context>
    auto call(Operation operation, H handler, Context ctx = io_result::use_sync) {
        return makeAsyncCall(operation, adapted(std::move(handler)), ctx);
    }
};

template <typename T>
Handler::OnSuccess withDefaultHttpWrap(T h) {
    return [h = std::move(h)](yhttp::response resp) {
        const unsigned status = resp.status;
        if (helpers::successCode(status)) {
            if constexpr (std::is_same_v<std::invoke_result_t<T, yhttp::response>, Result>) {
                return h(std::move(resp));
            } else {
                h(std::move(resp));
                return Result::success;
            }
        } else {
            return helpers::retriableCode(status) ? Result::retry : Result::fail;
        }
    };
}

RequestBuilder<Method::Get, false> toGET(const Endpoint& e);
RequestBuilder<Method::Post, false> toPOST(const Endpoint& e);

struct Client {
    http::headers headersToPass_;
    TvmManager::TicketsPtr tickets_;
    RequestStatsPtr lai_;
    AsyncRun run_;

    Client(TvmManager::TicketsPtr tickets, http::headers headersToPass,
           AsyncRun run, RequestStatsPtr stats)
        : headersToPass_(std::move(headersToPass))
        , tickets_(std::move(tickets))
        , lai_(std::move(stats))
        , run_(std::move(run))
    { }

    template<class Builder>
    auto pressure(Builder&& b, const std::string& tvm) const {
        if (tickets_) {
            helpers::setTickets(b, tvm, tickets_);
        }
        return b.headers("hdrs"_hdr=headersToPass_);
    }

    template<class Builder>
    auto req(Builder builder) const {
        return std::make_shared<RequestLoop<Builder>>(std::move(builder), run_, lai_);
    }

    auto toGET(const Endpoint& e) const {
        return pressure(::http_getter::toGET(e), e.tvmService());
    }

    auto toPOST(const Endpoint& e) const {
        return pressure(::http_getter::toPOST(e), e.tvmService());
    }
};

using ClientPtr = std::shared_ptr<Client>;

}
