#pragma once

#include <mail/http_getter/client/include/request_status.h>
#include <mail/http_getter/client/include/metrics.h>
#include <mail_errors/error_code.h>
#include <http_getter/http_request.h>


namespace http_getter {

struct Handler {
    using OnSuccess = std::function<Result(yhttp::response)>;
    using OnError = std::function<void(boost::system::error_code)>;
    OnError error;
    OnSuccess success;
};

using AsyncRun = std::function<void(Request, CallbackType)>;
using TypedToken = std::function<void(mail_errors::error_code)>;
using TypedRun = std::function<void(Request, Handler, TypedToken)>;
using TypedRunFactory = std::function<TypedRun(TypedHttpClientPtr, RequestStatsPtr, std::string)>;

template<class H>
inline Handler adapted(H h) {
    return Handler {
        .error=[] (auto) { },
        .success=std::move(h)
    };
}

template<>
inline Handler adapted<Handler>(Handler h) {
    return h;
}

using CompletionType = yhttp::simple_call::callback_type;
using Executor = TypedHttpClientPtr;


template<class Operation>
inline std::string getOperationName(Operation operation) {
    if constexpr (std::is_same_v<Operation, std::string> || std::is_same_v<Operation, const char*>) {
        return operation;
    } else {
        return std::string{yamail::data::reflection::to_string(operation)};
    }
}

enum class Errors {
    UnexpectedException = 1,
};

class ErrorCategory: public boost::system::error_category {
public:
    const char* name() const noexcept {
        return "http_category";
    }

    std::string message(int error) const {
        switch(Errors(error)) {
            case Errors::UnexpectedException: return "unexpected exception";
        }
        return "unknown error";
    }
};

const ErrorCategory& getErrorCategory();

inline mail_errors::error_code::base_type make_error_code(Errors e) {
    return mail_errors::error_code::base_type(static_cast<int>(e), getErrorCategory());
}

}

namespace boost::system {
template <>
struct is_error_code_enum<http_getter::Errors> {
    static const bool value = true;
};
}
