#pragma once

#include <string>
#include <chrono>
#include <optional>
#include <mail/http_getter/client/include/exception.h>
#include <mail/http_getter/client/include/fmt.h>


namespace http_getter {

struct Endpoint {
    struct Timeout {
        std::chrono::milliseconds connect = std::chrono::seconds{1};
        std::chrono::milliseconds total = std::chrono::seconds{1};
    };

    struct Data {
        std::string url;
        std::optional<std::string> fallback;
        std::string method;
        std::string tvm_service;
        bool keep_alive = false;
        bool log_response_body = false;
        bool log_post_body = false;
        Timeout timeout_ms;
        unsigned tries = 0;
    };

private:
    std::optional<Data> data_;

    Data& data();

    Endpoint& operator=(const Endpoint&) = delete;
    Endpoint& operator=(Endpoint&&) = delete;

    void dataConstraint() const;
    void fixDoubleSlash();
    void checkAndFixData();

    void checkDataIsNotEmpty() const;

public:
    Endpoint(Data&& raw);

    Endpoint() = default;
    Endpoint(const Endpoint& other) = default;
    Endpoint(Endpoint&& other) = default;

    void setData(Data&& d);
    void setData(const Data& d);

    const Data& data() const;
    bool empty() const;

    template<class ... Args>
    Endpoint format(const Args&... args) const {
        Data dd = data();
        dd.method = fmt::format(dd.method, args...);

        return Endpoint(std::move(dd));
    }

    const std::string& url() const {
        return data().url;
    }
    const std::optional<std::string>& fallback() const {
        return data().fallback;
    }

    const std::string& method() const {
        return data().method;
    }
    const std::string& tvmService() const {
        return data().tvm_service;
    }

    bool keepAlive() const {
        return data().keep_alive;
    }
    bool logResponseBody() const {
        return data().log_response_body;
    }
    bool logPostBody() const {
        return data().log_post_body;
    }

    std::chrono::milliseconds totalTimeout() const {
        return data().timeout_ms.total;
    }
    std::chrono::milliseconds connectTimeout() const {
        return data().timeout_ms.connect;
    }

    unsigned tries() const {
        return data().tries;
    }
};

std::ostream& operator<<(std::ostream& out, const Endpoint& e);

}
