#pragma once

#include <mail/http_getter/client/include/typed_metrics.h>
#include <boost/asio/spawn.hpp>


namespace http_getter {

struct Interpreter {
    Interpreter(Handler handler, TypedToken ctx, RequestStatsPtr lai, std::string operationName, bool logResponseBody)
        : handler_{std::move(handler)}
        , ctx_{std::move(ctx)}
        , metrics_{detail::TypedMetrics::create(std::move(lai), std::move(operationName), logResponseBody)}
    {
        metrics_->starting();
    }

    template <typename ResultHandler>
    void operator()(yhttp::response res, ResultHandler&& resHandler) {
        auto action = Result::retry;
        const auto bodyToLog = metrics_->getBodyToLog(res);
        const auto httpCode = static_cast<unsigned>(res.status);
        try {
            action = handler_.success(std::move(res));
            metrics_->responseIsGot(httpCode, bodyToLog);
        } catch (const boost::coroutines::detail::forced_unwind&) {
            throw;
        } catch (const boost::system::system_error& ex) {
            handler_.error(ex.code());
            metrics_->errorCodeIsGot(ex.code(), bodyToLog);
        } catch (const std::exception& ex) {
            handler_.error(
                boost::system::error_code(
                    static_cast<int>(Errors::UnexpectedException),
                    getErrorCategory()
                )
            );
            metrics_->exceptionIsGot(ex, bodyToLog);
        }
        metrics_->setLastAction(action);
        resHandler(convert(action), yhttp::response{});
    }

    void operator()(boost::system::error_code ec, yhttp::response) {
        if (ec) {
            handler_.error(ec);
            metrics_->errorCodeIsGot(ec, std::nullopt);
        }
        ctx_(mail_errors::error_code());
        metrics_->finishing();
    }

private:
    static ymod_httpclient::response_status convert(Result r) {
        switch(r) {
            case Result::success: return ymod_httpclient::response_status::ok;
            case Result::retry: return ymod_httpclient::response_status::tmp_error;
            case Result::fail: return ymod_httpclient::response_status::perm_error;
        }
    }

    Handler handler_;
    TypedToken ctx_;
    detail::TypedMetricsPtr metrics_;
};

}
