#pragma once

#include <mail/http_getter/client/include/common.h>
#include <mail/http_getter/client/include/metrics.h>


namespace http_getter::detail {

struct TypedMetrics {

    template <typename... Args>
    static std::shared_ptr<TypedMetrics> create(Args&&... args) {

        struct make_shared_enabler: TypedMetrics {
            make_shared_enabler(RequestStatsPtr lai, std::string operationName, bool logResponseBody)
                : TypedMetrics(std::move(lai), std::move(operationName), logResponseBody) {}
        };

        return std::make_shared<make_shared_enabler>(std::forward<Args>(args)...);
    }

    void starting() {
        if (!lai_) {
            return;
        }
        startTime_ = std::chrono::steady_clock::now();
        lai_->starting(operationName_);
    }

    void finishing() {
        if (!lai_) {
            return;
        }
        lai_->finishing(operationName_, lastAction_);
    }

    void errorCodeIsGot(const boost::system::error_code& ec, const std::optional<std::string>& bodyToLog) {
        if (!lai_) {
            return;
        }
        lai_->responseIsGot(operationName_, tryNumber_, getElapsed(), bodyToLog, ec);
    }

    void exceptionIsGot(const std::exception& ex, const std::optional<std::string>& bodyToLog) {
        if (!lai_) {
            return;
        }
        lai_->responseIsGot(operationName_, tryNumber_, getElapsed(), bodyToLog, ex);
    }

    void responseIsGot(const unsigned httpCode, const std::optional<std::string>& bodyToLog) {
        if (!lai_) {
            return;
        }
        lai_->responseIsGot(operationName_, tryNumber_, getElapsed(), *bodyToLog, httpCode, lastAction_);
    }

    std::optional<std::string> getBodyToLog(const yhttp::response& res) const {
        if (!lai_) {
            return std::nullopt;
        }
        return logResponseBody_ ? res.body
                                : makeHiddenValue(res.body);
    }

    void setLastAction(Result action) {
        if (!lai_) {
            return;
        }
        lastAction_ = action;
        tryNumber_++;
    }

private:

    TypedMetrics(const TypedMetrics&) = delete;
    TypedMetrics& operator=(const TypedMetrics&) = delete;
    TypedMetrics(RequestStatsPtr lai, std::string operationName, bool logResponseBody)
        : lai_{std::move(lai)}
        , operationName_{std::move(operationName)}
        , logResponseBody_{logResponseBody}
        , tryNumber_{0}
        , lastAction_{Result::retry}
    {}

    RequestStatsPtr lai_;
    std::string operationName_;
    bool logResponseBody_;

    std::chrono::steady_clock::time_point startTime_;
    size_t tryNumber_;
    Result lastAction_;

    double getElapsed() const {
        const auto milliseconds = std::chrono::duration_cast<std::chrono::milliseconds>(
            std::chrono::steady_clock::now() - startTime_
        );
        return static_cast<double>(milliseconds.count()) / 1000.;
    }

};

using TypedMetricsPtr = std::shared_ptr<TypedMetrics>;

}
