#pragma once

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <mail/http_getter/client/include/client.h>
#include <yamail/expected.h>

namespace http_getter {

inline auto createDummy(AsyncRun run) {
    return std::make_shared<Client>(nullptr, http::headers(), run, nullptr);
}

inline auto createDummy() {
    return createDummy([](Request, CallbackType cb) {
        cb(boost::system::error_code(), yhttp::response());
    });
}

inline auto createDummy(yhttp::response resp) {
    return createDummy([=](Request, CallbackType cb) {
        cb(boost::system::error_code(), resp);
    });
}

struct ResponseSequence {
    MOCK_METHOD(yhttp::response, get, (), (const));
    MOCK_METHOD(yhttp::response, get, (Request), (const));
};
using ResponseSequencePtr = std::shared_ptr<ResponseSequence>;


inline auto createDummy(ResponseSequencePtr responses) {
    return createDummy([=](Request, CallbackType cb) {
        cb(boost::system::error_code(), responses->get());
    });
}

inline auto createDummyWithRequest(ResponseSequencePtr responses) {
    return createDummy([=](Request r, CallbackType cb) {
        cb(boost::system::error_code(), responses->get(std::move(r)));
    });
}


inline auto createTypedDummy(TypedRunFactory factory) {
    return std::make_shared<TypedClient>(nullptr, http::headers(), std::move(factory), nullptr);
}

using TypedDummyResponse = yamail::expected<yhttp::response, boost::system::error_code>;

inline auto createTypedDummy(TypedDummyResponse resp) {
    auto factory = [=](TypedHttpClientPtr, RequestStatsPtr, std::string) -> TypedRun {
        return [=](Request, Handler handler, TypedToken token) {
            if (resp) {
                handler.success(std::move(resp).value());
            } else {
                handler.error(resp.error());
            }
            token(boost::system::error_code{});
        };
    };
    return createTypedDummy(std::move(factory));
}

inline auto createTypedDummy(ResponseSequencePtr responses) {
    auto factory = [=](TypedHttpClientPtr, RequestStatsPtr, std::string) -> TypedRun {
        return [=](Request, Handler handler, TypedToken token) {
            handler.success(responses->get());
            token(boost::system::error_code{});
        };
    };
    return createTypedDummy(std::move(factory));
}

inline auto createTypedDummyWithRequest(ResponseSequencePtr responses) {
    auto factory = [=](TypedHttpClientPtr, RequestStatsPtr, std::string) -> TypedRun {
        return [=](Request r, Handler handler, TypedToken token) {
            handler.success(responses->get(std::move(r)));
            token(boost::system::error_code{});
        };
    };
    return createTypedDummy(std::move(factory));
}

}
