#include <mail/http_getter/client/include/module.h>
#include <mail/http_getter/client/include/logger.h>
#include <mail/http_getter/client/include/interpreter.h>
#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <ymod_tvm/module.h>


namespace http_getter {

namespace {
template<class T>
void checkConfigExists(const std::string& name) {
    if (!yplatform::exists<T>(name)) {
        throw std::runtime_error("Cannot find config for module "+name);
    }
}

inline const std::string& reqOrUniqId(const yplatform::task_context_ptr& ctx) {
    return ctx->request_id().empty() ? ctx->uniq_id() : ctx->request_id();
}

}

struct RequestTaskContext: public yplatform::task_context {
    using yplatform::task_context::task_context;

    const std::string& get_name() const override {
        static const std::string name = "http_getter";
        return name;
    }
};

struct BaseClient {
    void initBase(ModuleLogger log, const yplatform::ptree& cfg) {
        tvm_ = std::make_shared<TvmManager>(parseTvmManagerConfig(cfg.get_child("tvm")));

        auto headersToPass = cfg.equal_range("headers_to_pass");
        for (auto it = headersToPass.first; it != headersToPass.second; ++it) {
            headersToPass_.push_back(boost::algorithm::to_lower_copy(it->second.get<std::string>("name")));
        }

        httpLog_ = cfg.get<std::string>("logger.http");
        uniqIdHeaderName_ = boost::algorithm::to_lower_copy(cfg.get<std::string>("uniq_id_header_name"));

        const std::string tvm = cfg.get<std::string>("dependencies.tvm");
        checkConfigExists<ymod_tvm::tvm2_module>(tvm);
        auto module = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>(tvm);

        for(const std::string& sercice: tvm_->tvm2ServicesWithServiceTicket()) {
            module->subscribe_service_ticket(sercice,
                [tvm = tvm_, log] (const auto& ec, const auto& s, const auto& t) {
                if (ec) {
                    LOGDOG_(log, error, log::message="error while updating service ticket: " + s, log::error_code=ec);
                } else if (!tvm->updateTicket(s, t)) {
                    LOGDOG_(log, error, log::message="cannot update ticket for service: " + s);
                }
            });
        }
    }

    http::headers makeHeaders(const std::string& uniqId) const {
        http::headers h;
        h.add(uniqIdHeaderName_, uniqId);

        return h;
    }

    template<class Type, class RunMaker>
    auto make(const ymod_webserver::request& request, RequestStatsPtr stats, const RunMaker& maker) const {
        http::headers h;
        for (const std::string& name: headersToPass_) {
            if (const auto it = request.headers.find(name); it != request.headers.end()) {
                h.add(name, it->second);
            }
        }

        std::string uniqId;
        std::string user;
        if (const auto it = request.headers.find(uniqIdHeaderName_); it != request.headers.end()) {
            uniqId = it->second;
        }
        h.add(uniqIdHeaderName_, uniqId);

        if (const auto it = request.headers.find("x-ya-user-ticket"); it != request.headers.end()) {
            user = it->second;
        }
        const auto& ctx = request.ctx();
        return std::make_shared<Type>(
            tvm_->tickets(user), std::move(h), maker(ctx), std::move(stats)
        );
    }

    TvmPtr tvm_;
    std::vector<std::string> headersToPass_;
    std::string httpLog_;
    std::string tvmModuleName_;
    std::string uniqIdHeaderName_;
};

struct SimpleClient: public ClientModule, public yplatform::module, public BaseClient {
    virtual ~SimpleClient() = default;

    void init(const yplatform::ptree& cfg) {
        ModuleLogger log = getModuleLogger(cfg.get<std::string>("logger.module"));

        initBase(log, cfg);

        const std::string http = cfg.get<std::string>("dependencies.http");
        checkConfigExists<yhttp::simple_call>(http);
        call_ = yplatform::find<yhttp::simple_call, std::shared_ptr>(http);

        LOGDOG_(log, notice, log::message="http_getter::SimpleClient module initialized");
    }

    Logger httpLogger(const std::string& uid, const std::string& requestId) const override {
        return getHttpLogger(requestId, uid, httpLog_);
    }

    AsyncRun asyncRunTask(const yplatform::task_context_ptr& task) const {
        return [c = call_, task] (Request req, CallbackType cb) {
            ::http_getter::asyncRun(*c, task, req, cb);
        };
    }

    AsyncRun asyncRun(const std::string& id) const {
        auto task = boost::make_shared<RequestTaskContext>("", id);
        return asyncRunTask(task);
    }

    ClientPtr create(const std::string& uniqId, RequestStatsPtr stats) const override {
        return std::make_shared<Client>(
            tvm_->tickets(""), makeHeaders(uniqId), asyncRun(uniqId), std::move(stats)
        );
    }

    ClientPtr create(const ymod_webserver::request& request, RequestStatsPtr stats) const override {
        return this->make<Client>(request, stats, [this] (const auto& id) { return this->asyncRunTask(id); } );
    }

    ClientPtr create(const yplatform::task_context_ptr& ctx, RequestStatsPtr stats) const override {
        return std::make_shared<Client>(
            tvm_->tickets(""), makeHeaders(reqOrUniqId(ctx)), asyncRunTask(ctx), std::move(stats)
        );
    }

    std::shared_ptr<yhttp::simple_call> call_;
};

struct TypedModule: public TypedClientModule, public yplatform::module, public BaseClient {
    virtual ~TypedModule() = default;

    void init(const yplatform::ptree& cfg) {
        ModuleLogger log = getModuleLogger(cfg.get<std::string>("logger.module"));

        initBase(log, cfg);

        LOGDOG_(log, notice, log::message="http_getter::TypedClient module initialized");
    }

    Logger httpLogger(const std::string& uid, const std::string& requestId) const override {
        return getHttpLogger(requestId, uid, httpLog_);
    }

    static TypedRunFactory asyncRunTask(const yplatform::task_context_ptr& task) {
        return [task] (TypedHttpClientPtr call, RequestStatsPtr stats, std::string name) {
            return [c = std::move(call), task, stats, name=std::move(name)] (Request req, Handler handler, TypedToken token) {
                bool logResponseBody = req.options.logResponseBody.value_or(false);
                ::http_getter::asyncRun(*c, task, req, Interpreter(handler, token, stats, name, logResponseBody));
            };
        };
    }

    static TypedRunFactory asyncRun(const std::string& id) {
        auto task = boost::make_shared<RequestTaskContext>("", id);
        return asyncRunTask(task);
    }

    TypedClientPtr create(const std::string& uniqId, RequestStatsPtr stats) const override {
        return std::make_shared<TypedClient>(
            tvm_->tickets(""), makeHeaders(uniqId), asyncRun(uniqId), stats
        );
    }

    TypedClientPtr create(const ymod_webserver::request& request, RequestStatsPtr stats) const override {
        return this->make<TypedClient>(request, stats, &TypedModule::asyncRunTask);
    }

    TypedClientPtr create(const yplatform::task_context_ptr& ctx, RequestStatsPtr stats) const override {
        return std::make_shared<TypedClient>(
            tvm_->tickets(""), makeHeaders(reqOrUniqId(ctx)), asyncRunTask(ctx), std::move(stats)
        );
    }
};

}

DEFINE_SERVICE_OBJECT(http_getter::SimpleClient)
DEFINE_SERVICE_OBJECT(http_getter::TypedModule)
