#include <mail/http_getter/client/include/tvm.h>
#include <yamail/data/reflection/reflection.h>
#include <yamail/data/deserialization/ptree.h>


namespace http_getter {

TvmManager::Options parseTvmManagerConfig(const yplatform::ptree& node) {
    TvmManager::Options ret;

    auto services = node.equal_range("services");
    for (auto i = services.first; i != services.second; i++) {
        const std::string name = i->second.get("name", "");

        if (name.empty()) {
            throw TvmException("http::parseOptions: name of service is empty");
        }

        try {
            TvmManager::Option opt;
            opt.service = i->second.get<bool>("service");
            opt.user = i->second.get<bool>("user");

            ret[name] = opt;
        } catch (const std::exception& ex) {
            std::ostringstream out;
            out << "http::parseOptions: cannot parse tvm config, name=" << name
                << " what=" << ex.what();
            throw TvmException(out.str());
        }
    }

    return ret;
}

TvmManager::TvmManager(TvmManager::Options options)
    : options_(std::make_shared<TvmManager::Options>(std::move(options)))
    , tickets_(std::make_shared<TvmTickets>())
{ }

TvmManager::TicketsPtr TvmManager::tickets(std::string user) const {
    return std::make_shared<Tickets>(std::move(user), tickets_, options_);
}

std::vector<std::string> TvmManager::tvm2ServicesWithServiceTicket() const {
    std::vector<std::string> ret;
    for(const auto& opt : *options_) {
        if (opt.second.service) {
            ret.push_back(opt.first);
        }
    }
    return ret;
}

bool TvmManager::updateTicket(const std::string& service, const std::string& ticket) {
    const auto opt = options_->find(service);

    if (opt == options_->end() || !(opt->second.service)) {
        return false;
    } else {
        std::lock_guard<yplatform::spinlock> guard(lock_);

        auto newTickets = std::make_shared<TvmTickets>(*tickets_);
        (*newTickets)[service] = ticket;
        tickets_ = newTickets;

        return true;
    }
}

bool TvmManager::Tickets::enabled(const std::string& service, bool TvmManager::Option::*option) const {
    auto opt = options_->find(service);

    if (opt == options_->end()) {
        throw TvmException(std::string("cannot find options for service: ") + service);
    }

    return opt->second.*option;
}

TvmManager::Tickets::Tickets(std::string user, TvmTicketsPtr tickets, OptionsPtr options)
    : user_(std::move(user))
    , serviceTickets_(std::move(tickets))
    , options_(std::move(options))
{ }

std::optional<std::string> TvmManager::Tickets::user(const std::string& service) const {
    return enabled(service, &Option::user) ? std::make_optional(user_) : std::nullopt;
}

std::optional<std::string> TvmManager::Tickets::service(const std::string& service) const {
    if (enabled(service, &Option::service)) {
        const auto it = serviceTickets_->find(service);

        if (it == serviceTickets_->end()) {
            throw TvmException(std::string("cannot get service ticket for service: ") + service);
        } else {
            return std::make_optional(it->second);
        }
    }

    return std::nullopt;
}

}
