#pragma once

#include <optional>
#include <boost/hana/string.hpp>

namespace http_getter::detail {

enum class Type { Header, Argument };

using std::optional;
namespace hana = boost::hana;

template <typename Key, typename Value>
using attribute = hana::pair<Key, Value>;


template <typename Key, typename Value>
constexpr decltype(auto) name(const attribute<Key, Value>&) {
    return hana::to<const char*>(Key::key);
}

template <typename Key, typename Value>
constexpr decltype(auto) value(const attribute<Key, Value>& v) {
    return hana::second(v);
}

template <typename Key, typename Value>
inline decltype(auto) makeAttribute(const Key& key, Value&& v) {
    return hana::make_pair(key, std::forward<Value>(v));
}

template <Type type, char ...name>
struct attributeDefinition {
    static constexpr auto key = hana::to<const char*>(hana::string_c<name...>);

    template <typename T>
    decltype(auto) operator = (const T v) const {
        return makeAttribute(*this, std::move(v));
    }
};

namespace operators {
template <typename CharT, CharT ...str>
constexpr attributeDefinition<Type::Argument, str...> operator"" _arg() {
    return {};
}

template <typename CharT, CharT ...str>
constexpr attributeDefinition<Type::Header, str...> operator"" _hdr() {
    return {};
}
}

using namespace operators;

template<typename T>
struct isArgument : std::false_type {};

template <char ...name>
struct isArgument<attributeDefinition<Type::Argument, name...>> : std::true_type {};


template<typename T>
struct isHeader : std::false_type {};

template <char ...name>
struct isHeader<attributeDefinition<Type::Header, name...>> : std::true_type {};

template <typename Key, typename Value>
void checkArgument(const attribute<Key, Value>&) {
    static_assert(isArgument<Key>::value, "aguments must have literal _arg");
}

template <typename Key, typename Value>
void checkHeader(const attribute<Key, Value>&) {
    static_assert(isHeader<Key>::value, "aguments must have literal _hdr");
}

} // namespace http_getter::detail
