#pragma once

#include "attribute.h"
#include "error.h"
#include "options.h"
#include <butil/http/url.h>
#include <butil/http/arguments.h>
#include <butil/http/headers.h>
#include <ymod_httpclient/call.h>

namespace http_getter::detail {

using namespace http;

struct Request {
    Request(yhttp::request&& request, Options&& options)
        : request(std::move(request))
        , options(std::move(options))
    { }
    yhttp::request request;
    Options options;
};

enum class Method {
    Get,
    Head,
    Post,
    Put,
    Delete,
};

template<Method m>
concept MethodHasBody = (m == Method::Post) || (m == Method::Put);

constexpr auto requestId = "X-Request-Id"_hdr;
constexpr auto userTicket = "X-Ya-User-Ticket"_hdr;
constexpr auto serviceTicket = "X-Ya-Service-Ticket"_hdr;

template <typename T>
void merge(T& map, const T& insert) {
    for (const auto& [key, value] : insert) {
        if (const auto p = map.find(key); p != map.end() && p->second != value) {
            using namespace std::literals;
            throw Error("cannot merge maps with common element "s + key + " in forming request"s);
        }
    }
    map.insert(insert.begin(), insert.end());
}

inline void mergeHeaders(http::headers& map, const http::headers& insert) {
    merge(map.headers, insert.headers);
}

inline void mergeArguments(HttpArguments& map, const HttpArguments& insert) {
    merge(map.arguments, insert.arguments);
}

template <typename Arg>
struct addHeader {
    template <typename T>
    static auto apply(http::headers& headersMap, const T& v) {
        mergeHeaders(headersMap, {{{name(v), {value(v)}}}});
    }
};

template <>
struct addHeader <const http::headers&> {
    template <typename T>
    static auto apply(http::headers& headersMap, const T& v) {
        mergeHeaders(headersMap, value(v));
    }
};

template <typename Arg>
struct addGetArg {
    template <typename T>
    static void apply(HttpArguments& getArgsMap, const T& v) {
        mergeArguments(getArgsMap, {{{name(v), {value(v)}}}});
    }

    template <typename Key, typename Value>
    static void apply(HttpArguments& getArgsMap, const attribute<Key, std::optional<Value>>& v) {
        if (value(v)) {
            mergeArguments(getArgsMap, {{{name(v), {*value(v)}}}});
        }
    }

    template <typename Key>
    static void apply(HttpArguments&, const attribute<Key, std::nullopt_t>&) {
    }
};

template <>
struct addGetArg <const HttpArguments&> {
    template <typename T>
    static void apply(HttpArguments& getArgsMap, const T& v) {
        mergeArguments(getArgsMap, value(v));
    }
};

template <Method m, bool bodyIsSet = false, typename Parent = std::monostate>
struct RequestBuilder : public Parent {
    static inline constexpr Method method = m;

    using BodyPtr = boost::shared_ptr<std::string>;

    RequestBuilder(std::string&& url)
        : Parent{}
        , url_(std::move(url))
        , body_(boost::make_shared<std::string>())
    { }

    template<class T, class L>
    decltype(auto) timeouts(const std::chrono::duration<T, L>& total,
                        const std::chrono::duration<T, L>& connect) {
        timeouts_.total = yplatform::time_traits::duration_cast<yplatform::time_traits::duration>(total);
        timeouts_.connect = yplatform::time_traits::duration_cast<yplatform::time_traits::duration>(connect);
        return *this;
    }

    #define SETTER(type, name) \
        RequestBuilder& name(type v) & { this->name##_ = std::move(v); return *this; }\
        RequestBuilder&& name(type v) && { this->name##_ = std::move(v); return static_cast<RequestBuilder&&>(*this); }\
        RequestBuilder name(type v) const & { auto retval = *this; retval.name##_ = std::move(v); return retval; }

    SETTER(bool, keepAlive)
    SETTER(bool, logHeaders)
    SETTER(bool, logPostArgs)
    SETTER(unsigned, maxAttempts)
    SETTER(url, fallback)
    SETTER(bool, logResponseBody)
    SETTER(::ymod_httpclient::timeouts, timeouts)

    #undef SETTER

    decltype(auto) postArgs(const HttpArguments& args) {
        return RequestBuilder<method, true>(
            *this, boost::make_shared<std::string>(args.format())
        );
    }

    template <typename Body>
    decltype(auto) body(Body&& body) {
        BodyPtr b;
        if constexpr (std::is_same_v<std::remove_reference_t<Body>, BodyPtr>) {
            b = std::forward<Body>(body);
        } else {
            b = boost::make_shared<std::string>(std::forward<Body>(body));
        }

        return RequestBuilder<method, true, Parent>(
            *this, std::move(b)
        );
    }

    template<typename ...Args>
    decltype(auto) getArgs(Args&& ...args) {
        (checkArgument(args), ...);
        (addGetArg<decltype(value(args))>::apply(getArgs_, args), ...);
        return *this;
    }

    template<typename ...Args>
    decltype(auto) headers(Args&& ...args) {
        (checkHeader(args), ...);
        (addHeader<decltype(value(args))>::apply(headers_, args), ...);
        return *this;
    }

    Request primary() {
        return Request(request(url_), options());
    }

    Request fallback() {
        if (!fallback_) {
            throw std::runtime_error("there is no fallback in RequestBuilder");
        }
        return Request(request(*fallback_), options());
    }

    Request make() {
        return Request(request(url_), std::move(*this).options());
    }

    bool hasFallback() const {
        return static_cast<bool>(fallback_);
    }

private:
    url url_;
    std::optional<url> fallback_;
    http::headers headers_;
    HttpArguments getArgs_;
    BodyPtr body_;
    std::optional<bool> keepAlive_;
    std::optional<bool> logHeaders_;
    std::optional<bool> logPostArgs_;
    ::ymod_httpclient::timeouts timeouts_;
    std::optional<unsigned> maxAttempts_;
    std::optional<bool> logResponseBody_;

    template<Method mm> requires MethodHasBody<mm>
    RequestBuilder(RequestBuilder<mm, false, Parent>& another, BodyPtr&& body)
        : Parent(another)
        , url_(std::move(another.url_))
        , fallback_(std::move(another.fallback_))
        , headers_(std::move(another.headers_))
        , getArgs_(std::move(another.getArgs_))
        , body_(std::move(body))
        , keepAlive_(std::move(another.keepAlive_))
        , logHeaders_(std::move(another.logHeaders_))
        , logPostArgs_(std::move(another.logPostArgs_))
        , timeouts_(std::move(another.timeouts_))
        , maxAttempts_(std::move(another.maxAttempts_))
        , logResponseBody_(std::move(another.logResponseBody_))
    { }

    Options options() && {
        return Options {
            .logPostArgs=std::move(logPostArgs_),
            .logHeaders=std::move(logHeaders_),
            .keepAlive=std::move(keepAlive_),
            .timeouts=std::move(timeouts_),
            .maxAttempts=std::move(maxAttempts_),
            .logResponseBody=std::move(logResponseBody_)
        };
    }

    Options options() & {
        return Options {
            .logPostArgs=logPostArgs_,
            .logHeaders=logHeaders_,
            .keepAlive=keepAlive_,
            .timeouts=timeouts_,
            .maxAttempts=maxAttempts_,
            .logResponseBody=logResponseBody_
        };
    }

    yhttp::request request(const url& u) {
        std::string url = u.urlPart() + combineArgs(u.source(), getArgs_);

        if constexpr (MethodHasBody<method>) {
            return yhttp::request {
                .method= (method == Method::Post) ?
                    yhttp::request::method_t::POST :
                    yhttp::request::method_t::PUT,
                .url=std::move(url),
                .headers=headers_.format(),
                .body=body_,
                .attempt=0
            };
        } else if constexpr (method == Method::Head) {
            return yhttp::request::HEAD(std::move(url), headers_.format());
        } else if constexpr (method == Method::Get) {
            return yhttp::request::GET(std::move(url), headers_.format());
        } else if constexpr (method == Method::Delete) {
            return yhttp::request::DELETE(std::move(url), headers_.format());
        }
    }

    template<Method, bool, typename>
    friend struct RequestBuilder;
};

inline auto post(std::string url) {
    return RequestBuilder<Method::Post>(std::move(url));
}

inline auto get(std::string url) {
    return RequestBuilder<Method::Get>(std::move(url));
}

inline auto head(std::string url) {
    return RequestBuilder<Method::Head>(std::move(url));
}

} // namespace http_getter::detail
