import click
import yaml
import asyncio
import ujson
import uvloop

from datetime import timedelta
from pathlib import Path
from aiohttp import web
from aiohttp.web_response import Response
from yaml import SafeLoader, MappingNode
from typing import Union, Mapping

from mail.pg.huskydb.settings import HuskyDbSettings

from mail.python.theatre.stages.base import Stage
from mail.python.theatre.app.settings import LogSettings
from mail.python.theatre.app.settings.base import SettingsNode
from mail.python.theatre.stages.db_stats.roles.director import Director as DbStatsDirector
from mail.python.theatre.stages.db_stats.types import Db, DbHost, TierRunPolicy, DbSignal


class Settings(SettingsNode):
    db = HuskyDbSettings()
    log = LogSettings()


def load_config_file(config_path: str):
    def timedelta_yaml_constructor(loader: SafeLoader, node: MappingNode) -> timedelta:
        return timedelta(**loader.construct_mapping(node))

    def tier_run_policy_constructor(loader, node):
        return TierRunPolicy(loader.construct_scalar(node))

    yaml.add_constructor('!TimeDelta', timedelta_yaml_constructor, Loader=yaml.SafeLoader)
    yaml.add_constructor('!TierRunPolicy', tier_run_policy_constructor, Loader=yaml.SafeLoader)

    with open(config_path, 'r') as f:
        config = yaml.safe_load(f)
    return config


def make_db_stat_director(settings: Settings, config: Mapping[str, Union[str, Mapping]]) -> DbStatsDirector:
    db = Db(hosts=[DbHost(settings.db.pg_dsn())])
    signals = [
        DbSignal(**signal)
        for signal in config['signals']
    ]
    return DbStatsDirector([db], signals, config['db_host_status_poll_time'], fatPollersConfig=None)


class DbStatsStage(Stage):
    def __init__(self, env: str, config_path: str):
        self.config = load_config_file(config_path)
        settings = Settings
        settings.log.dir_path = Path(self.config['log']['dir'])
        settings.log.level = self.config['log']['level']
        super().__init__(env=env, name='husky-dbstat', settings=Settings)

    async def create_app(self) -> web.Application:
        director = make_db_stat_director(self.settings, self.config)
        self.app.update(director=director)

        async def unistat_handler(_):
            return Response(text=ujson.dumps(self.app['director'].stats), content_type='application/json')

        async def shutdown(_):
            await director.stop()

        await director.start()

        self.app.router.add_get('/unistat', unistat_handler)
        self.app.router.add_get('/ping', lambda _: Response(text='pong\n'))
        self.app.on_shutdown.append(shutdown)

        return self.app


@click.command()
@click.option('--host', default='::0', help='IP addr to bind to')
@click.option('--port', default=80, help='HTTP port to listen')
@click.option('--config', 'config_file', help='Configuration file')
@click.option('--env', help='Environment name')
def main(host, port, config_file, env):
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())

    stage = DbStatsStage(env, config_file)
    stage.run(host=host, port=port)
