import os

import requests
from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import SubprocessComponent, WithPort
from mail.devpack.lib.components.fakebb import FakeBlackbox
from mail.devpack.lib.components.mulcagate import Mulcagate
from mail.devpack.lib.components.sharddb import ShardDb
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.tvm_mulcagate import TvmMulcagate
from mail.devpack.lib.components.tvmtool import TvmTool
from mail.york.devpack.components.york import York
from mail.pg.huskydb.devpack.components.huskydb import HuskyDb
from ora2pg.app import config_file

from library.python import resource


class HuskyApi(SubprocessComponent, WithPort):
    NAME = "husky-api"
    DEPS = [HuskyDb, ShardDb, FakeBlackbox, Sharpei, Mulcagate, York, TvmTool, TvmMulcagate]

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(HuskyApi, cls).gen_config(port_generator, config=config)
        return dict(
            tvm={
                "client_id": next(port_generator),
                "src_client_comp_names": ['tvm_client'],
            },
            **base
        )

    def __init__(self, env, components):
        super(HuskyApi, self).__init__(env, components)
        self.tvm_client_id = self.config[self.NAME]['tvm']['client_id']
        self.bin_file = env.get_arcadia_bin('mail/husky/bin/api/husky-api')

    @property
    def app_config_path(self):
        return os.path.join(self.etc_path, "app.yaml")

    @property
    def env_config_path(self):
        return os.path.join(self.etc_path, "env.yaml")

    @staticmethod
    def get_devpack_config(components):
        filename = config_file.env_to_config_file('devpack')
        return config_file.read_config_file(filename).format(
            blackbox=components[FakeBlackbox].url,
            sharpei=components[Sharpei].api().location,
            huskydb_dsn=components[HuskyDb].dsn(),
            mulcagate=components[Mulcagate].host,
            mulcagate_port=components[Mulcagate].port,
            mulcagate_ca_path=components[Mulcagate].ssl_cert_path,
            york=components[York].yhttp.url,
        )

    def init_root(self):
        helpers.mkdir_recursive(self.etc_path)
        helpers.mkdir_recursive(self.log_dir)
        helpers.mkdir_recursive(self.pid_dir)

        env_config = self.get_devpack_config(self.components)
        helpers.write2file(env_config, self.env_config_path)
        app_config = resource.find('resfs/file/mail/husky/devpack/husky-api.yaml').decode('utf-8').format(
            env_config_path=self.env_config_path,
            tvmtool_url="http://localhost:{}".format(self.components[TvmTool].port),
            tvm_local_token=self.components[TvmTool].AUTHTOKEN,
            tvm_client_id=self.tvm_client_id,
            tvm_src_client_ids=[self.tvm_client_id],
        )
        helpers.write2file(app_config, self.app_config_path)

    @property
    def pid_dir(self):
        return os.path.join(self.root, 'var', 'run')

    def start(self):
        cmd = [
            self.bin_file,
            '--config-path', self.app_config_path,
            '--port', str(self.port),
            '--work-dir', self.root,
            '--daemon',
        ]
        with self.starting(cmd):
            helpers.wait_ping(self.logger, self.ping, "pong")
            helpers.wait_ping(self.logger, self.pingdb, "ok")

    def info(self):
        return {
            "root": self.root,
            "state": self.state,
            "bin_file": self.bin_file,
            "app_config": self.app_config_path,
            "env_config": self.env_config_path,
            "port": self.port,
        }

    @property
    def host_port(self):
        return 'localhost:{}'.format(self.port)

    @property
    def url(self):
        return "http://{}".format(self.host_port)

    def _signed_request(self, uri, method='get', timeout=10, **kwargs):
        headers = kwargs.pop('headers', {})
        headers['X-Ya-Service-Ticket'] = self.components[TvmTool].get_ticket(
            src=self.tvm_client_id,
            dst=self.tvm_client_id,
        )
        url = 'http://{}/{}'.format(self.host_port, uri.lstrip('/'))
        self.logger.info('%s request, target: %s, kwargs: %s', method.upper(), url, kwargs)
        return requests.request(
            method,
            url=url,
            headers=headers,
            timeout=timeout,
            **kwargs
        )

    def get(self, target, **kwargs):
        return self._signed_request(uri=target, method='get', **kwargs)

    def post(self, target, **kwargs):
        return self._signed_request(uri=target, method='post', **kwargs)

    def ping(self):
        return self.get('/ping')

    def pingdb(self):
        return self.get('/pingdb')

    def get_clone_user_task(self, source_uid, dest_uid):
        return self.get('/clone_user/{source_uid}/into/{dest_uid}'.format(source_uid=source_uid, dest_uid=dest_uid))

    def create_clone_user_task(self, source_uid, dest_uid, with_deleted_box=False, task_args={}):
        uri = '/clone_user/{source_uid}/into/{dest_uid}'.format(source_uid=source_uid, dest_uid=dest_uid)
        if with_deleted_box:
            uri += '/with_deleted_box'

        return self.post(
            uri,
            headers={"Content-Type": "application/json"},
            json=task_args
        )

    def create_delete_user_task(self, uid, right_now=False):
        uri = '/delete_user/{uid}'.format(uid=uid)
        if right_now:
            uri += '/right_now'
        return self.post(uri)
