import os

from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import SubprocessComponent, WithPort
from mail.devpack.lib.components.fakebb import FakeBlackbox
from mail.devpack.lib.components.mulcagate import Mulcagate
from mail.devpack.lib.components.sharddb import ShardDb
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.tvm_mulcagate import TvmMulcagate
from mail.devpack.lib.components.tvmtool import TvmTool
from mail.york.devpack.components.york import York
from mail.pg.huskydb.devpack.components.huskydb import HuskyDb
from ora2pg.app import config_file

from library.python import resource


class Husky(SubprocessComponent, WithPort):
    NAME = "husky"
    DEPS = [FakeBlackbox, HuskyDb, Mulcagate, Sharpei, ShardDb, York, TvmMulcagate]

    def __init__(self, env, components):
        super(Husky, self).__init__(env, components)
        self.bin_file = env.get_arcadia_bin('mail/husky/bin/worker/husky-worker')
        self.mds_id = env.config['husky']['mds_id']
        self.tvm_id = env.config['husky']['tvm']['client_id']

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(Husky, cls).gen_config(port_generator, config=config)
        return dict(
            tvm={
                "client_id": next(port_generator),
            },
            mds_id=config['tvm_mulcagate']['tvm']['client_id'],
            **base
        )

    @property
    def app_config_path(self):
        return os.path.join(self.etc_path, "app.yaml")

    @property
    def env_config_path(self):
        return os.path.join(self.etc_path, "env.yaml")

    @staticmethod
    def get_devpack_config(components):
        filename = config_file.env_to_config_file('devpack')
        return config_file.read_config_file(filename).format(
            blackbox=components[FakeBlackbox].url,
            sharpei=components[Sharpei].api().location,
            huskydb_dsn=components[HuskyDb].dsn(),
            mulcagate=components[TvmMulcagate].host,
            mulcagate_port=components[TvmMulcagate].port,
            mulcagate_ca_path=components[TvmMulcagate].ssl_cert_path,
            york=components[York].yhttp.url,
            client_id=components[Husky].tvm_id,
            mds_id=components[Husky].mds_id,
        )

    def init_root(self):
        helpers.mkdir_recursive(self.etc_path)
        helpers.mkdir_recursive(self.log_dir)

        env_config = self.get_devpack_config(self.components)
        helpers.write2file(env_config, self.env_config_path)
        app_config = resource.find('resfs/file/mail/husky/devpack/husky.yaml').decode('utf-8').format(
            env_config_path=self.env_config_path,
            husky_name=self.name,
            bb_tvm_id=self.components[FakeBlackbox].tvm_id,
            mds_id=self.components[TvmMulcagate].tvm_id,
            tvm_id=self.components[Husky].tvm_id,
            tvm_daemon_url="http://localhost:{}".format(self.components[TvmTool].port),
            tvm_local_token=self.components[TvmTool].AUTHTOKEN,
        )
        helpers.write2file(app_config, self.app_config_path)

    def start(self):
        cmd = [
            self.bin_file,
            '--env', 'devpack',
            '--app-config-path', self.app_config_path,
            '--env-config-path', self.env_config_path,
            '--port', str(self.port),
        ]
        env = {
            'DEVPACK_HUSKYDB_PORT': str(self.components[HuskyDb].port()),
        }
        with self.starting(cmd, env) as proc:
            # TODO :: check ping, maybe check worker status in huskydb
            import time
            time.sleep(5)
            if not proc.is_running():
                raise RuntimeError('%s is not started (failed after start)' % self.name)

    def info(self):
        return {
            "root": self.root,
            "state": self.state,
            "bin_file": self.bin_file,
            "app_config": self.app_config_path,
            "env_config": self.env_config_path,
            "port": self.port
        }
