import argparse
import getpass
import gunicorn.app.base

from ora2pg.app.config_file import get_config
from .api import create_app
from .config import API_CONFIG_ITEMS


class HuskyGunicornApp(gunicorn.app.base.BaseApplication):
    def __init__(self, config_path, options=None):
        self.options = options or {}
        config = get_config(config_path, API_CONFIG_ITEMS)
        if 'log_filepath' not in config and 'errorlog' in options:
            config['log_filepath'] = options['errorlog']
        self.application = create_app(config)
        super(HuskyGunicornApp, self).__init__()

    def load_config(self):
        config = {
            key: value for key, value in self.options.items()
            if key in self.cfg.settings and value is not None
        }
        for key, value in config.items():
            self.cfg.set(key.lower(), value)

    def load(self):
        return self.application


def main():
    parser = argparse.ArgumentParser(
        'husky-api',
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )
    parser.add_argument(
        '--config-path',
        help="Path to husky-api config file"
    )
    parser.add_argument(
        '--work-dir',
        default='.',
        help="Workdir to run husky-api process into",
    )
    parser.add_argument(
        '-p', '--port',
        type=int,
        default=8080,
        help="HTTP port to listen to",
    )
    parser.add_argument(
        '-u', '--run_as',
        default=getpass.getuser(),
        help="Unix username",
    )
    parser.add_argument(
        '-w', '--workers',
        type=int,
        default=1,
        help='Worker count',
    )
    parser.add_argument(
        '--pidfile',
        help='Path to pidfile',
        default='var/run/husky-api.pid',
    )
    parser.add_argument(
        '--logfile',
        help='Path to logfile',
        default='var/log/husky-api/husky-api.log',
    )
    parser.add_argument(
        '-d', '--daemon',
        help='Daemonize',
        action='store_true'
    )
    args = parser.parse_args()
    options = {
        'name': 'husky-api',
        'bind': '[::1]:%d' % args.port,
        'user': args.run_as,
        'workers': args.workers,
        'pidfile': args.pidfile,
        'errorlog': args.logfile,
        'capture_output': True,
        'chdir': args.work_dir,
        'daemon': args.daemon,
        'capture-output': True,
    }
    HuskyGunicornApp(args.config_path, options).run()
