# coding: utf-8
"""Working loop for husky subprocesses"""

from .tasks import get_handler
from .tasks.errors import NotSupportedError, ArgumentError
from .types import ResultData, Errors

import sys
import os
import traceback
import logging
log = logging.getLogger(__name__)


def error_info_from_tb(message, tb):
    """Make some easy oneline error info to put in database"""

    (filename, lineno, funcname, codestr) = traceback.extract_tb(tb)[-1]
    return '{filename}:{lineno}, in {funcname}:{codestr} {message}'.format(**locals())


def success_result(transfer_id):
    return ResultData(transfer_id=transfer_id, error=Errors.NoError, error_message=None, task_output=None)


def error_result(transfer_id, e=Errors.Unknown, err_msg=None):
    if err_msg is None:
        _, exc_value, tb = sys.exc_info()
        if exc_value is not None and tb is not None:
            err_msg = error_info_from_tb(str(exc_value), tb)
        else:
            err_msg = ''
    return ResultData(transfer_id=transfer_id, error=e, error_message=err_msg, task_output=None)


class StopRequestedError(RuntimeError):
    pass


class Worker(object):
    def __init__(self, app, worker_id, data_q, resp_q):
        self.worker_id = worker_id
        self.app = app
        self.data_q = data_q
        self.resp_q = resp_q
        try:
            self.tvm = app.args.tvm
        except AttributeError:
            self.tvm = None
        log.info('Worker #%d is ready to work!', self.worker_id)

    def dispatch(self, transfer_id, userdata):
        try:
            TaskHandler = get_handler(userdata.task)
            task_handler = TaskHandler(
                app=self.app,
                transfer_id=transfer_id,
                uid=userdata.uid,
                task_args=userdata.task_args,
            )
            log.info('Worker #%d gonna go with task id=%d',
                     self.worker_id,
                     transfer_id)
            return task_handler.run() or success_result(transfer_id)
        except NotSupportedError as e:
            return ResultData(
                transfer_id=transfer_id,
                error=Errors.NotSupported,
                error_message=repr(e),
                task_output=None,
            )
        except ArgumentError as e:
            return ResultData(
                transfer_id=transfer_id,
                error=Errors.WrongArgs,
                error_message=repr(e),
                task_output=None,
            )

    def process(self):
        try:
            userdata = self.data_q.get()
        except BaseException as e:
            log.exception(e)
            raise StopRequestedError()

        if userdata is None:
            log.info(
                'Worker #%d recieved None from queue and shutting down',
                self.worker_id
            )
            raise StopRequestedError()
        transfer_id = userdata.transfer_id
        try:
            self.app.set_context(userdata)
            result = self.dispatch(transfer_id, userdata)
            if result.error == Errors.NoError:
                log.info(
                    'Worker #%d succeeded with transfer_id=%d',
                    self.worker_id,
                    transfer_id
                )
            self.resp_q.put(result)
        except BaseException as e:
            log.exception(
                'Worker #%d failed with transfer_id=%d : %r',
                self.worker_id,
                transfer_id,
                e
            )
            self.resp_q.put(error_result(transfer_id))
        finally:
            self.app.unset_context()


def main(app, data_q, resp_q):
    """Subprocess loop"""

    worker_id = os.getpid()
    worker = Worker(**locals())
    try:
        while True:
            worker.process()
    except StopRequestedError:
        return
