from dateutil.parser import parse as date_parse

from mail.pypg.pypg.common import transaction

from mail.husky.husky.api_queries import ApiQueries
from .base import BaseTask, get_tvm_ticket
from .mixin import ShardResolverMixin
from .errors import NotSupportedError
from mail.husky.husky.types import Task
from ora2pg.sharpei import get_pg_dsn_from_sharpei
from ora2pg.blackbox import get_all_by_uid
from ora2pg.clone_user import CloneConfig, clone_user, NotSupportedUserError
from ora2pg.restore_deleted_box import restore_deleted_box
from ora2pg.restore_hidden_trash import restore_hidden_trash
from ora2pg.tools.find_master_helpers import find_huskydb, find_sharddb


class CloneUser(BaseTask, ShardResolverMixin):
    name = Task.CloneUser
    required_args = ['dest_shard', 'source_user_uid']

    @property
    def source_user_uid(self):
        return self.task_args['source_user_uid']

    @property
    def dest_user_uid(self):
        return self.uid

    @property
    def huskydb_dsn(self):
        if not hasattr(self, '_huskydb_dsn'):
            self._huskydb_dsn = find_huskydb(self.config)
        return self._huskydb_dsn

    @property
    def dest_shard_id(self):
        return self.resolve_endpoint(str(self.task_args['dest_shard'])).shard_id

    @property
    def loaded_shard_id(self):
        return self.dest_shard_id

    def _get_date_arg(self, name):
        dt = self.get_arg(name)
        return dt and date_parse(dt)

    @property
    def min_received_date(self):
        return self._get_date_arg('min_received_date')

    @property
    def max_received_date(self):
        return self._get_date_arg('max_received_date')

    def mark_clone_audit_as_completed(self):
        with transaction(self.huskydb_dsn) as conn:
            q = ApiQueries(conn)
            q.mark_clone_audit_as_completed(
                source_uid=self.source_user_uid,
                dest_uid=self.dest_user_uid,
            )

    def restore_deleted_box(self):
        maildb_dsn = get_pg_dsn_from_sharpei(
            self.config.sharpei,
            self.dest_user_uid,
            self.config.maildb_dsn_suffix
        )
        restore_deleted_box(maildb_dsn, self.dest_user_uid)
        restore_hidden_trash(maildb_dsn, self.dest_user_uid)

    def run(self):
        try:
            self.config.mulcagate.tvm_ticket = self.app.args.tvm.get(self.app.args.mds_id)
        except AttributeError:
            self.config.mulcagate.tvm_ticket = None

        config = CloneConfig(
            dest_shard_id=self.dest_shard_id,
            sharddb=find_sharddb(self.config),
            sharpei=self.config.sharpei,
            mailhost=self.config.mailhost,
            blackbox=self.config.blackbox,
            mulcagate=self.config.mulcagate,
            maildb_dsn_suffix=self.config.maildb_dsn_suffix,
            huskydb=self.huskydb_dsn,
        )

        bb_tvm_ticket = get_tvm_ticket(self.config.tvm, self.config.bb_tvm_id)
        try:
            clone_user(
                source_user=get_all_by_uid(self.config.blackbox, self.source_user_uid, bb_tvm_ticket),
                dest_user=get_all_by_uid(self.config.blackbox, self.uid, bb_tvm_ticket),
                config=config,
                force=self.task_args.get('force', False),
                min_received_date=self.min_received_date,
                max_received_date=self.max_received_date,
            )
            if self.task_args.get('restore_deleted_box', False):
                self.restore_deleted_box()
        except NotSupportedUserError as e:
            raise NotSupportedError(str(e))

        self.mark_clone_audit_as_completed()
