import logging

import dateutil.parser
from mail.pypg.pypg.common import transaction
from mail.pypg.pypg.query_handler import ExpectOneItemError

from mail.husky.husky.types import Task, ResultData, Errors
from mail.husky.husky.sharddb import SharddbAdaptor, GetDeletedUserShardIdError
from ora2pg.sharpei import get_connstring_by_id
from pymdb.operations import DeleteUser as DeleteUserOperation
from pymdb.queries import Queries
from .base import BaseTask

log = logging.getLogger(__name__)


class DeleteMailUser(BaseTask):
    name = Task.DeleteMailUser
    required_args = ['deleted_date']

    @property
    def deleted_date(self):
        return dateutil.parser.parse(self.task_args['deleted_date'])

    def get_maildb_dsn(self, shard_id):
        return get_connstring_by_id(
            sharpei=self.config.sharpei,
            shard_id=shard_id,
            dsn_suffix=self.config.maildb_dsn_suffix,
        )

    def run(self):
        shard_id = self.get_deleted_user_shard_id()
        if shard_id is None:
            return ResultData(
                transfer_id=self.transfer_id,
                error=Errors.NoSuchUser,
                error_message='User not found in shards.deleted_user',
                task_output=None,
            )

        with transaction(self.get_maildb_dsn(shard_id)) as conn:
            queries = Queries(conn, self.uid)
            try:
                user = queries.user()
            except ExpectOneItemError:
                log.warning(
                    'User with uid %r does not exist in shard_id %r',
                    self.uid,
                    shard_id,
                )
                return
            if not user.is_here:
                return ResultData(
                    transfer_id=self.transfer_id,
                    error=Errors.NotSupported,
                    error_message='Try delete a user which is not here',
                    task_output=None,
                )

            DeleteUserOperation(conn, self.uid)(self.deleted_date)

    def get_deleted_user_shard_id(self):
        try:
            return SharddbAdaptor(self.app.args).get_deleted_user_shard_id(self.uid)
        except GetDeletedUserShardIdError:
            return None
