from __future__ import unicode_literals

from datetime import datetime
from dateutil.parser import parse as parse_dt
from ora2pg.tools.find_master_helpers import find_huskydb

from mail.pypg.pypg import common as pg
from mail.pypg.pypg.query_handler import ExpectOneItemError
from ora2pg import sharpei
from pymdb.operations import StoreMessage
from pymdb.types import StoreCoordinates, MailishCoordinates, StoreHeaders, StoreThreading, StoreRecipient, \
    StoreMimePart
from pymdb.queries import Queries
from mail.husky.husky.types import Task

from .base import BaseTask
from .queries import TaskQueries
from .errors import ArgumentError


class PutLetter(BaseTask):
    """
    Таск для покладки произвольного письма
    """
    name = Task.PutLetter
    required_args = ['global_args_id']

    global_args = {}

    @property
    def global_args_id(self):
        return self.task_args['global_args_id']

    def _get_letter_args(self, arg_id):
        with pg.readonly_repeatable_read_transaction(dsn=find_huskydb(self.config)) as husky_conn:
            try:
                return TaskQueries(husky_conn).get_global_args(args_id=arg_id).task_args
            except ExpectOneItemError:
                raise ArgumentError('Letter args not found')

    def _letter_args(self):
        arg_id = self.global_args_id
        if arg_id not in PutLetter.global_args:
            PutLetter.global_args[arg_id] = self._get_letter_args(arg_id)
        return PutLetter.global_args[arg_id]

    def _build_params(self, mail_conn, letter_args):
        folder = Queries(mail_conn, self.uid).folder_by_type(letter_args['folder_type'])

        return {
            "coords": StoreCoordinates(
                fid=folder.fid,
                tid=None,
                seen=False,
                deleted=False,
                received_date=datetime.now(),
                size=letter_args['message_size'],
                st_id=letter_args['message_st_id'],
                pop_uidl=[],
                attributes=[],
            ),
            "headers": StoreHeaders(
                subject=letter_args['message_subject'],
                firstline=letter_args['message_first_line'],
                hdr_date=parse_dt(letter_args['message_date']),
                hdr_message_id=(letter_args['message_id']),
                extra_data=""
            ),
            "recipients": [
                StoreRecipient(type='from', name=letter_args['message_from_name'],
                               email=letter_args['message_from_email'])
            ],
            "threads_meta": StoreThreading(
                rule="hash",
                references_hashes=[],
                in_reply_to_hash=None,
                hash_value=0,
                hash_namespace="subject",
                hash_uniq_key=0,
                sort_options=""
            ),
            "attaches": [],
            "mime": [StoreMimePart(**m) for m in letter_args['mime']],
            "lids": [],
            "mailish_coords": MailishCoordinates(
                imap_id=None,
                imap_time=None
            ),
            "quiet": False
        }

    def run(self):
        user_dsn = sharpei.get_pg_dsn_from_sharpei(sharpei=self.config.sharpei, uid=self.uid,
                                                   dsn_suffix=self.config.maildb_dsn_suffix)

        letter_args = self._letter_args()

        with pg.transaction(dsn=user_dsn) as mail_conn:
            try:
                params = self._build_params(mail_conn, letter_args)
            except KeyError as e:
                raise ArgumentError('Bad letter args: %s' % e)

            operation = StoreMessage(mail_conn, self.uid)
            operation(**params)
            operation.commit()
