from __future__ import print_function

import json
import logging

from ora2pg.app.config_file import get_config
from mail.python.tvm_requests import Tvm

from .base import get_handler as get_task_handler, task_handlers
from ora2pg.app.arguments import Argument, Flag, Uid
from ora2pg.app.logs import logged
from ora2pg.app.transfer_app import parse_arguments, make_app_from_env


def get_tvm_from_conf(config):
    return Tvm(
        tvm_daemon_url=config['tvm'].get('daemon_url', 'http://localhost:1'),
        client_id=config['tvm']['client_id'],
        local_token=config['tvm'].get('local_token', None),
    )


@logged
def main():
    args = parse_arguments(
        __doc__,
        [
            Argument(
                '--task',
                'Task name',
                choices=sorted(task_handlers().keys()),
                required=True,
            ),
            Argument(
                '--transfer-id',
                'Transfer id (to be used for subtasks)',
                type_=int,
                default=100500,
            ),
            Uid,
            Argument(
                '--task-args',
                'Task arguments, json',
                type_=json.loads,
                default={},
            ),
            Argument(
                '--config-path',
                'Path to husky config, can be repeated',
            ),
            Flag(
                '--ipdb',
                'Enable post-mortem ipdb session',
            ),
        ]
    )
    if args.ipdb:
        import sys
        from IPython.core import ultratb
        sys.excepthook = ultratb.FormattedTB(mode='Verbose', color_scheme='Linux', call_pdb=1)
    config = get_config(args.config_path)
    config.update(args._asdict())
    config['log_level'] = logging.DEBUG

    tvm = get_tvm_from_conf(config)
    config.pop('tvm')
    if 'mulcagate_ca_path' not in config.keys():
        config['mulcagate_ca_path'] = None

    app = make_app_from_env(tvm=tvm, **config)
    task_handler = get_task_handler(app.args.task)
    print(task_handler(
        app=app,
        transfer_id=app.args.transfer_id,
        uid=app.args.uid,
        task_args=app.args.task_args,
    ).run())
