import logging

from dateutil.parser import parse as date_parse

from mail.husky.husky.types import Task, ResultData, Errors
from mail.husky.husky.api_queries import ApiQueries
from mail.pypg.pypg.common import transaction
from ora2pg.blackbox import BBInfo, get_all_by_uid, NonExistentUserError
from ora2pg.sharpei import get_connstring_by_id
from ora2pg.transfer import transfer, UserNotInFromDb, UserBlocked
from ora2pg.duplicate import duplicate_user
from ora2pg.sync_user import sync_user, presync_user
from ora2pg.tools.find_master_helpers import find_huskydb, get_huskydb_pooled_conn
from pymdb.queries import Queries
from .base import BaseTask, get_tvm_ticket
from .mixin import ShardResolverMixin
from .errors import ArgumentError

log = logging.getLogger(__name__)


def _has_too_many_messages(uid, shard_id, app_args):
    conn_str = get_connstring_by_id(app_args.sharpei, shard_id, app_args.maildb_dsn_suffix)
    with transaction(conn_str) as conn:
        queries = Queries(conn, uid)
        message_count = queries.message_count()
        assert len(message_count) == 1
        return message_count[0]['count'] > app_args.message_count_limit


class Transfer(BaseTask, ShardResolverMixin):
    """Call core transfer and process certain errors"""

    required_args = ['to_db', 'from_db']
    name = Task.Transfer

    @property
    def from_db_endpoint(self):
        return self.resolve_endpoint(self.task_args['from_db'].encode('utf-8'))

    @property
    def to_db_endpoint(self):
        return self.resolve_endpoint(self.task_args['to_db'].encode('utf-8'))

    @property
    def loaded_shard_id(self):
        return self.to_db_endpoint.shard_id

    @property
    def only_duplicate(self):
        return self.get_arg('duplicate', False)

    @property
    def can_transfer_deleted(self):
        return self.get_arg('force', False)

    @property
    def min_received_date(self):
        min_rcvd = self.get_arg('min_received_date')
        return min_rcvd and date_parse(min_rcvd)

    @property
    def presync(self):
        if self.get_arg('sync', False) and self.get_arg('presync', False):
            raise ArgumentError("presync and sync arguments can't be set to true at the same time")
        return self.get_arg('presync', False)

    @property
    def sync(self):
        if self.get_arg('sync', False) and self.get_arg('presync', False):
            raise ArgumentError("presync and sync arguments can't be set to true at the same time")
        return self.get_arg('sync', False)

    @property
    def presync_received_date(self):
        presync_rcvd = self.get_arg('presync_received_date')
        return presync_rcvd and date_parse(presync_rcvd)

    @property
    def can_transfer(self):
        dsn = find_huskydb(self.app.args)
        with get_huskydb_pooled_conn(dsn, autocommit=False) as conn:
            q = ApiQueries(conn)
            try:
                shards = q.get_shiva_shard(shard_id=self.loaded_shard_id)
                if shards and not shards[0].can_transfer_to:
                    return False
            except Exception as e:
                log.exception(e)
                return False
            return True

    def user_has_too_many_messages(self):
        return _has_too_many_messages(self.uid, self.from_db_endpoint.shard_id, self.app.args)

    def run_duplicate(self, user):
        duplicate_user(
            app=self.app,
            user=user,
            to_db=self.to_db_endpoint,
            from_db=self.from_db_endpoint,
        )

    def run_presync(self, user):
        presync_user(
            app=self.app,
            user=user,
            to_db=self.to_db_endpoint,
            from_db=self.from_db_endpoint,
            presync_received_date=self.presync_received_date,
        )

    def run_sync(self, user):
        sync_user(
            app=self.app,
            user=user,
            to_db=self.to_db_endpoint,
            from_db=self.from_db_endpoint,
            presync_received_date=self.presync_received_date,
        )

    def run(self):
        user = BBInfo(uid=self.uid, login=None, suid=None, default_email=None)
        if not self.can_transfer:
            return ResultData(
                transfer_id=self.transfer_id,
                error=Errors.ClosedShard,
                error_message='destination shard closed for transfer',
                task_output=None,
            )
        if not self.can_transfer_deleted:
            try:
                user = get_all_by_uid(self.config.blackbox, self.uid, get_tvm_ticket(self.config.tvm, self.config.bb_tvm_id))
            except NonExistentUserError as exc:
                return ResultData(
                    transfer_id=self.transfer_id,
                    error=Errors.NoSuchUser,
                    error_message=str(exc),
                    task_output=None,
                )

        if self.only_duplicate:
            return self.run_duplicate(user)

        if self.presync:
            return self.run_presync(user)

        if self.sync:
            return self.run_sync(user)

        try:
            if self.get_arg('check_message_count', True) and self.user_has_too_many_messages():
                error_msg = 'Transfer of uid={uid} is deffered \'cause it has too many messages'.format(uid=self.uid)
                log.info(error_msg)
                return ResultData(
                    transfer_id=self.transfer_id,
                    error=Errors.DeferredDueToMessageCountLimit,
                    error_message=error_msg,
                    task_output=None
                )
        except (AssertionError, KeyError) as e:
            log.exception('Failed to get message count for uid=%d: %s', self.uid, e)
            return ResultData(
                transfer_id=self.transfer_id,
                error=Errors.FailedGetData,
                error_message=repr(e),
                task_output=None,
            )

        try:
            transfer(
                app=self.app,
                user=user,
                to_db=self.to_db_endpoint,
                from_db=self.from_db_endpoint,
                fill_change_log=self.get_arg('fill_change_log', False),
                change_tabs=self.get_arg('change_tabs', True),
                force=self.can_transfer_deleted,
                firstline_options=self.get_arg('firstline_options', None),
                min_received_date=self.min_received_date,
            )
        except UserNotInFromDb as e:
            log.exception(e)
            return ResultData(
                transfer_id=self.transfer_id,
                error=Errors.InvalidFromDb,
                error_message=repr(e),
                task_output=None,
            )
        except UserBlocked as e:
            log.exception(e)
            return ResultData(
                transfer_id=self.transfer_id,
                error=Errors.UserBlocked,
                error_message=repr(e),
                task_output=None,
            )
