import logging

from mail.husky.husky.types import Task
from mail.pypg.pypg.common import qexec, transaction
from mail.pypg.pypg.query_conf import Query
from ora2pg import copy_user
from ora2pg.blackbox import get_all_by_uid, NonExistentUserError
from ora2pg.cleanup import clean_user
from ora2pg.copy_user import Pg2Pg
from ora2pg.sharpei import get_connstring_by_id, update_shard_id, get_pg_dsn_from_sharpei
from ora2pg.tools.find_master_helpers import find_sharddb
from ora2pg.transfer_data import get_transfer_info, get_user_in_endpoint, DbEndpoint
from ora2pg.transfer_delete_queue import transfer_delete_queue
from .base import BaseTask, get_tvm_ticket
from .mixin import ShardResolverMixin

log = logging.getLogger(__name__)


class TransferFromBackup(BaseTask, ShardResolverMixin):
    """Call core transfer and process certain errors"""

    required_args = ['from_dsn', 'to_db']
    name = Task.TransferFromBackup

    @property
    def from_dsn(self):
        return self.task_args['from_dsn']

    @property
    def to_db_endpoint(self):
        return self.resolve_endpoint(self.task_args['to_db'].encode('utf-8'))

    def run(self):
        try:
            get_all_by_uid(self.config.blackbox, self.uid, get_tvm_ticket(self.config.tvm, self.config.bb_tvm_id))
        except NonExistentUserError:
            log.warning('User is not present in blackbox')

        self.check_dst(self.uid)
        self.copy_user(self.uid)

    def check_dst(self, uid):
        user_shard_dsn = get_pg_dsn_from_sharpei(self.app.args.sharpei, uid, self.app.args.maildb_dsn_suffix)
        with transaction(user_shard_dsn) as conn:
            cur = qexec(
                conn,
                Query('select * from mail.users where uid = %(uid)s', ['uid'], None),
                uid=uid,
            )
            if cur.rowcount:
                raise RuntimeError('Expect to have no user info in sharpei-provided shard, got: %s' % list(cur))

    def copy_user(self, uid):
        to_dsn = get_connstring_by_id(
            self.app.args.sharpei,
            self.to_db_endpoint.shard_id,
            self.app.args.maildb_dsn_suffix
        )
        transfer_info = get_transfer_info(
            src=get_user_in_endpoint(
                db_endpoint=DbEndpoint.make_pg(self.from_dsn),
                uid=uid,
            ),
            dst=get_user_in_endpoint(
                db_endpoint=self.to_db_endpoint,
                uid=uid,
            ),
        )
        sharddb = find_sharddb(self.app.args)

        with transaction(self.from_dsn) as from_conn:
            clean_user(uid, to_dsn)
            # assert_there_are_no_backup_firstlines(from_pg_conn, uid)
            copy_user.pg2pg(
                from_conn,
                to_dsn,
                transfer_info,
                fill_changelog=True,
                # options.fill_change_log,
                # options.tabs_mapping,
                # options.firstline_options,
                Copier=Pg2PgFromBackup,
            )
            # with transaction(self.to_db_endpoint.dsn) as to_conn:
            #     cur = qexec(
            #         to_conn,
            #         Query('update mail.users set is_here = TRUE where uid = %(uid)s and is_here = FALSE', ['uid'], None),
            #         uid=uid
            #     )
            #     if not cur.rowcount:
            #         raise RuntimeError('Assert user present and not is_here')
            transfer_delete_queue(from_conn, to_dsn, uid, remove_from_source=False)
            # update_subs_status(to_dsn, uid)
            update_shard_id(sharddb, uid, self.to_db_endpoint.shard_id, is_deleted=False)


class Pg2PgFromBackup(Pg2Pg):
    def customize_to_user(self, to_user):
        to_user = super().customize_to_user(to_user)
        to_user.user.is_here = True
        return to_user
