from __future__ import unicode_literals

import psycopg2

from mail.husky.husky.types import Task
from ora2pg import sharpei
from ora2pg.tools.find_master_helpers import find_huskydb
from ora2pg.transfer_subscriptions import wait_for_normal_working_subs
from pymdb.operations import CreateFolder, CopyMessages, AddFolderToSubscribedFolders, \
    AddFolderToSharedFolders, AddSubscriberToSharedFolders, TransitSubscriptionState
from mail.pypg.pypg import common as pg
from .base import BaseTask
from .errors import ArgumentError


class TransferMarketCrm(BaseTask):
    """
    Таск для перевода ящика из SpaceCRM в SuiteCRM
    """
    name = Task.TransferMarketCrm
    required_args = ['sent_mids_table', 'source_user_uid', 'login']

    SENT_PROXY_UIDS = {
        'tcrm': 1120000000008946,
        'robot-tcrm-support': 1120000000038987,
    }
    RETRIES = [10] * 6 * 60  # 1 hour of tries

    @property
    def sent_mids_table(self):
        return self.task_args['sent_mids_table']

    @property
    def source_user_uid(self):
        return self.task_args['source_user_uid']

    @property
    def login(self):
        return self.task_args['login']

    def user_dsn(self, uid):
        return sharpei.get_pg_dsn_from_sharpei(
            sharpei=self.config.sharpei,
            uid=uid,
            dsn_suffix=self.config.maildb_dsn_suffix
        )

    def _get_sent_mids(self, table):
        with pg.readonly_repeatable_read_transaction(dsn=find_huskydb(self.config)) as husky_conn:
            try:
                cur = husky_conn.cursor()
                cur.execute('select email, array_agg(mid) from {table} group by 1'.format(table=table))
                result = dict(cur.fetchall())
            except psycopg2.errors.UndefinedTable:
                raise ArgumentError('Sent mids table not found')
        for email in result.keys():
            if email not in self.SENT_PROXY_UIDS:
                raise ArgumentError('Sent email is unknown to the task: %s' % email)
        return result

    def _make_sent_messages_folder(self, sent_proxy_uid, mids):
        with pg.transaction(self.user_dsn(sent_proxy_uid)) as conn:
            folder_name='sent-{}'.format(self.login)
            cur = conn.cursor()
            cur.execute('select fid, message_count from mail.folders where uid = %(uid)s and name = %(fname)s', dict(uid=sent_proxy_uid, fname=folder_name))
            result = cur.fetchall()
            if not result:
                create_op = CreateFolder(conn, sent_proxy_uid)(name=folder_name).commit()
                fid = create_op.fid
                message_count = 0
            else:
                fid, message_count = result[0]
            if not message_count:
                CopyMessages(conn, sent_proxy_uid)(mids=mids, dst_fid=fid).commit()
            return fid

    def _sync_folders(self, owner_uid, owner_fid, sub_uid, sub_fid):
        with pg.transaction(self.user_dsn(owner_uid)) as owner_conn:
            cur = owner_conn.cursor()
            cur.execute(
                '''
                select 1
                  from mail.shared_folder_subscriptions
                 where uid = %(uid)s
                   and fid = %(fid)s
                   and subscriber_uid = %(sub_uid)s
                ''',
                dict(uid=owner_uid, fid=owner_fid, sub_uid=sub_uid)
            )
            if cur.fetchall():
                wait_for_normal_working_subs(
                    owner_conn,
                    owner_uid,
                    retry_progression=self.RETRIES,
                )
                return
        with pg.transaction(self.user_dsn(sub_uid)) as sub_conn:
            AddFolderToSubscribedFolders(sub_conn, sub_uid)(
                fid=sub_fid,
                owner_uid=owner_uid,
                owner_fid=owner_fid,
            ).commit()
        with pg.transaction(self.user_dsn(owner_uid)) as owner_conn:
            AddFolderToSharedFolders(owner_conn, owner_uid)(fid=owner_fid).commit()
            AddSubscriberToSharedFolders(owner_conn, owner_uid)(
                fid=owner_fid,
                subscriber=sub_uid,
            ).commit()
        with pg.transaction(self.user_dsn(owner_uid)) as owner_conn:
            wait_for_normal_working_subs(
                owner_conn,
                owner_uid,
                retry_progression=self.RETRIES,
            )

    def _copy_sent_messages(self, uid, sent_proxy_uid, sent_proxy_fid):
        # Subscribe dest user to email proxy with sent
        with pg.transaction(self.user_dsn(uid)) as user_conn:
            cur = user_conn.cursor()
            cur.execute("select fid from mail.folders where uid = %(uid)s and type = 'sent'", dict(uid=uid))
            user_sent_fid = cur.fetchone()[0]
        self._sync_folders(
            owner_uid=sent_proxy_uid,
            owner_fid=sent_proxy_fid,
            sub_uid=uid,
            sub_fid=user_sent_fid,
        )
        # Remove subscription traces on dest user side
        with pg.transaction(self.user_dsn(uid)) as user_conn:
            cur = user_conn.cursor()
            cur.execute(
                """
                delete from mail.synced_messages
                 where uid = %(uid)s
                   and subscription_id IN (
                    select subscription_id
                      from mail.subscribed_folders
                     where uid = %(uid)s
                       and owner_uid = %(owner_uid)s
                       and owner_fid = %(owner_fid)s
                )
                """,
                dict(uid=uid, owner_uid=sent_proxy_uid, owner_fid=sent_proxy_fid),
            )
        # Terminate subscription on email proxy side via dobby
        with pg.transaction(self.user_dsn(sent_proxy_uid)) as proxy_conn:
            cur = proxy_conn.cursor()
            cur.execute(
                '''
                select subscription_id
                  from mail.shared_folder_subscriptions
                 where uid = %(uid)s
                   and fid = %(fid)s
                   and subscriber_uid = %(sub_uid)s
                ''',
                dict(uid=sent_proxy_uid, fid=sent_proxy_fid, sub_uid=uid)
            )
            sub_id = cur.fetchone()[0]
            TransitSubscriptionState(proxy_conn, sent_proxy_uid)(sub_id, 'unsubscription').commit()
        # Wait 'til subscription is terminated by dobby
        with pg.transaction(self.user_dsn(sent_proxy_uid)) as proxy_conn:
            wait_for_normal_working_subs(
                proxy_conn,
                sent_proxy_uid,
                retry_progression=(5, 10, 45, 60, 60, 120, 300, 600, 600, 600, 600, 600)
            )
        with pg.transaction(self.user_dsn(uid)) as user_conn:
            cur = user_conn.cursor()
            cur.execute(
                '''
                delete from mail.subscribed_folders
                 where uid = %(uid)s
                   and owner_uid = %(owner_uid)s
                   and owner_fid = %(owner_fid)s
                ''',
                dict(uid=uid, owner_uid=sent_proxy_uid, owner_fid=sent_proxy_fid),
            )

    def _inbox_fid(self, uid):
        with pg.transaction(self.user_dsn(uid)) as conn:
            cur = conn.cursor()
            cur.execute("select fid from mail.folders where type = 'inbox' and uid = %(uid)s", dict(uid=uid))
            return cur.fetchone()[0]

    def _subscribe_inboxes(self, source_uid, dest_uid):
        self._sync_folders(
            owner_uid=source_uid,
            owner_fid=self._inbox_fid(source_uid),
            sub_uid=dest_uid,
            sub_fid=self._inbox_fid(dest_uid),
        )

    def run(self):
        self._subscribe_inboxes(self.source_user_uid, self.uid)

        sent_mids = self._get_sent_mids(self.sent_mids_table)
        for email, mids in sent_mids.items():
            sent_proxy_uid = self.SENT_PROXY_UIDS[email]
            sent_proxy_fid = self._make_sent_messages_folder(sent_proxy_uid, mids)
            self._copy_sent_messages(self.uid, sent_proxy_uid, sent_proxy_fid)
